#include "set_lib.h"
#include <string.h>

int
set_deg(set a)
{
	/* Fast compute degree of a set... the number
	   of elements present in the set.  Assumes
	   that all word bits are used in the set
	   and that SETSIZE is a multiple of WORDSIZE.
	*/
	register word32 *p = &(a.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);
	register degree = 0;

	do {
		register word32 t = *p;

		while (t) {
			++degree;
			t &= (t - 1);
		}
	} while (++p < endp);

	return(degree);
}

set
set_or(set b,
set c)
{
	/* Fast set union operation */
	register word32 *p = &(b.setword[0]);
	register word32 *q = &(c.setword[0]);
	register word32 *endp = &(b.setword[SETWORDS]);

	do {
		*p |= *(q++);
	} while (++p < endp);
	return(b);
}

set
set_and(set b,
set c)
{
	/* Fast set intersection operation */
	register word32 *p = &(b.setword[0]);
	register word32 *q = &(c.setword[0]);
	register word32 *endp = &(b.setword[SETWORDS]);

	do {
		*p &= *(q++);
	} while (++p < endp);
	return(b);
}

set
set_dif(set b,
set c)
{
	/* Fast set difference operation */
	register word32 *p = &(b.setword[0]);
	register word32 *q = &(c.setword[0]);
	register word32 *endp = &(b.setword[SETWORDS]);

	do {
		*p &= (~ *(q++));
	} while (++p < endp);
	return(b);
}

set
set_of(int b)
{
	/* Fast singleton set constructor operation */
	static set a;

	memset(&a, 0, sizeof(a));
	if (b != nil) {
		a.setword[DIVWORD(b)] = BIT2WORD(MODWORD(b));
	}
	return(a);
}

set
set_not(set a)
{
	/* Fast not of set a (assumes all bits used) */
	register word32 *p = &(a.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);

	do {
		*p = (~ *p);
	} while (++p < endp);
	return(a);
}

int
set_equ(set a,
set b)
{
	/* Fast set equality comparison operation */
	register word32 *p = &(a.setword[0]);
	register word32 *q = &(b.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);

	do {
		if (*p != *q) return(0);
		++q;
	} while (++p < endp);
	return(1);
}

int
set_sub(set a,
set b)
{
	/* Fast check for a is a proper subset of b (alias a < b) */
	register word32 *p = &(a.setword[0]);
	register word32 *q = &(b.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);
	register int asubset = 0;

	do {
		/* Prune tests based on guess that most set words
		   will match, particularly if a is a subset of b.
		*/
		if (*p != *q) {
			if (*p & ~(*q)) {
				/* Fail -- a contains something b does not */
				return(0);
			}
			/* At least this word was a proper subset, hence
			   even if all other words are equal, a is a
			   proper subset of b.
			*/
			asubset = 1;
		}
		++q;
	} while (++p < endp);

	return(asubset);
}

int
set_int(set b)
{
	/* Fast pick any element of the set b */
	register word32 *p = &(b.setword[0]);
	register word32 *endp = &(b.setword[SETWORDS]);

	do {
		if (*p) {
			/* Found a non-empty word of the set */
			register int i = ((p - &(b.setword[0])) * WORDSIZE);
			register word32 t = 1;

			while (!(*p & t)) {
				++i; t += t;
			}
			return(i);
		}
	} while (++p < endp);

	/* Empty -- only element it contains is nil */
	return(nil);
}

set_el(int b,
set a)
{
	/* nil is an element of every set */
	if (b == nil) return(1);

	/* Otherwise, we have to check */
	return(a.setword[DIVWORD(b)] & BIT2WORD(MODWORD(b)));
}

set_nil(set a)
{
	/* Fast check for nil set */
	register word32 *p = &(a.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);

	/* The set is not empty if any word used to store
	   the set is non-zero.  This means one must be a
	   bit careful about doing things like negation.
	*/
	do {
		if (*p) return(0);
	} while (++p < endp);
	return(1);
}

char *
set_str(set a)
{
	/* Fast convert set a into ASCII char string...
	   assumes that all word bits are used in the set
	   and that SETSIZE is a multiple of WORDSIZE.
	   Trailing 0 bits are removed from the string.
	*/
	register word32 *p = &(a.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);
	static char str_tmp[SETSIZE+1];
	register char *q = &(str_tmp[0]);

	do {
		register word32 t = *p;
		register int i = 0;
		do {
			*(q++) = ((t & BIT2WORD(i)) ? '1' : '0');
		} while (++i < WORDSIZE);
	} while (++p < endp);

	/* Trim trailing 0s & NULL terminate the string */
	while ((q > &(str_tmp[1])) && (*(q-1) != '0')) --q;
	*q = 0;

	return(&(str_tmp[0]));
}

set
set_val(register char *s)
{
	/* Fast convert set ASCII char string into a set.
	   If the string ends early, the remaining set bits
	   are all made zero.
	*/
	static set a;
	register word32 *p = &(a.setword[0]);
	register word32 *endp = &(a.setword[SETWORDS]);

	do {
		register int i = 0;

		/* Start with a word with no bits on */
		*p = 0;
		do {
			if (*s) {
				if (*s == '1') {
					/* Turn-on this bit */
					*p |= BIT2WORD(i);
				}
				++s;
			}
		} while (++i < WORDSIZE);
	} while (++p < endp);

	return(a);
}
