/*	set_lib.h

	The following is a general-purpose set library which
	allows C programs to be written nearly as though the
	set data-type existed... sets are structs which can
	be freely copied by assignment statements, passed by
	value to functions, returned by value from functions,
	and allocated by ordinary declarations.

	Inconveniences:
	1. Sets are of fixed size because C cannot track a
	   runtime-sized array (no paramtypes yet).
	2. Sets cannot be compared without a function call.
	3. Some strangeness happens if only a portion of a
	   set is used or if there are bits in a word which
	   are ignored in the set representation; e.g., the
	   set_not routine may not function correctly.

	By H. Dietz
*/

#define	WORDSIZE 32			/* Usable bits per word */
#define	SETWORDS 4			/* Words per set */
#define	SETSIZE (WORDSIZE*SETWORDS)	/* Maximum items per set */
#define	MODWORD(x) ((x) & (WORDSIZE-1))	/* x % WORDSIZE */
#define	DIVWORD(x) ((x) >> 5)		/* x / WORDSIZE */
#define	BIT2WORD(x) (1 << (x))		/* 2 ** x */

#define	nil	SETSIZE			/* An impossible set member */

typedef	unsigned int	word32;
typedef	struct {
	word32 setword[SETWORDS];	/* Array of words to hold bits */
} set;

set	set_and(set b, set c);	/* returns arg1 intersection arg2 */
int	set_deg(set a);		/* returns degree (element count) of set arg */
set	set_dif(set b, set c);	/* returns set difference, arg1 - arg2 */
int	set_el(int b, set c);	/* returns non-0 if arg1 is an element of arg2 */
int	set_equ(set a, set b);	/* returns non-0 if arg1 equals arg2 */
int	set_int(set b);		/* returns an int which is in the set arg */
int	set_nil(set a);		/* returns non-0 if arg1 is nil */
set	set_not(set a);		/* returns not arg (well, sort-of) */
set	set_of(int b);		/* returns singleton set of int arg */
set	set_or(set b, set c);	/* returns arg1 union arg2 */
char	*set_str(set a);	/* formats a string representing set arg */
int	set_sub(set a, set b);	/* returns non-0 if arg1 is a proper subset of arg2 */
set	set_val(register char *s); /* converts set_str-format string arg into a set */
