/* ToDo:
	Add some type of test for bitwise shifts
	Add some type of test for fieldwise shifts and rotates
	Add tests for assignments and reductions
	Change handling of 32-bit tests to do fewer iterations and tests per
	  iteration.  Do a better job of this than I am now (e.g. take the
	  increments from the command line?)
	Add an option to allow the current iteration/test to be given on the
	  command line to allow manual checkpoint recovery.
*/

#undef NOTDEFD


/******** Support Functions **************************************************/
/* A macro describing a wrapper function in SWARC for most binary operations.
   The function arguments i and j are assigned to a pair of swar type
   variables, a and b, upon which the operation is performed.
   The result is stored in c, of the same type as a and b, from which
   the calling function may retrieve it.
*/
#define BINOP(name, op, bits, fields, sign, ms)			\
void name(ms sign##signed int i, ms sign##signed int j,		\
	  ms sign##signed int:##bits##[##fields##] c)		\
{								\
	ms sign##signed int:##bits##[##fields##] a;		\
	ms sign##signed int:##bits##[##fields##] b;		\
								\
	a = i;							\
	b = j;							\
								\
	c = a op b;						\
}


/* A macro describing a wrapper function in SWARC for binary shift operations.
   The function argument i is assigned to the swar type variable a.
   Currently, we don't handle vectorized shift counts, so we need to mask
   the second argument to a single byte; thus, the low byte of function
   argument j is assigned to the low byte of swar type variable b.
   The operation is performed on a and b, and the result is stored in c,
   of the same type as a and b, from which the calling function may
   retrieve it.
*/
#define BINOP_B(name, op, bits, fields, sign, ms)		\
void name(ms sign##signed int i, ms sign##signed int j, 	\
	  ms sign##signed int:##bits##[##fields##] c)		\
{								\
	ms sign##signed int:##bits##[##fields##] a;		\
	ms sign##signed int:##bits##[##fields##] b;		\
								\
	a = i;							\
	b = 0; b[0] = j;					\
								\
	c = a op b;						\
}



/* A macro to call the above macros to generate the wrapper functions for each
   of (at the moment, most of) the binary operators in the SWARC langage using
   unsigned or signed operands.
*/
#define DECLARE(bits, fields, uors, sign, mors, ms)			\
BINOP(add##bits##uors##mors,+, bits, fields, sign, ms)			\
BINOP(sub##bits##uors##mors,-, bits, fields, sign, ms)			\
BINOP(mul##bits##uors##mors,*, bits, fields, sign, ms)			\
BINOP(div##bits##uors##mors,/, bits, fields, sign, ms)			\
BINOP(mod##bits##uors##mors,%, bits, fields, sign, ms)			\
									\
/* Skip these for the moment...						\
BINOP_B(shl##bits##uors##mors,<<, bits, fields, sign, ms)		\
BINOP_B(shr##bits##uors##mors,>>, bits, fields, sign, ms)		\
*/									\
									\
BINOP(min##bits##uors##mors,?<, bits, fields, sign, ms)			\
BINOP(max##bits##uors##mors,?>, bits, fields, sign, ms)			\
BINOP(avg##bits##uors##mors,+/, bits, fields, sign, ms)			\
									\
BINOP(land##bits##uors##mors,&&, bits, fields, sign, ms)		\
BINOP(lor##bits##uors##mors,||, bits, fields, sign, ms)			\
									\
BINOP(and##bits##uors##mors,&, bits, fields, sign, ms)			\
BINOP(andn##bits##uors##mors,&~, bits, fields, sign, ms)		\
BINOP(or##bits##uors##mors,|, bits, fields, sign, ms)			\
BINOP(xor##bits##uors##mors,^, bits, fields, sign, ms)			\
									\
BINOP(eq##bits##uors##mors,==, bits, fields, sign, ms)			\
BINOP(ne##bits##uors##mors,!=, bits, fields, sign, ms)			\
BINOP(lt##bits##uors##mors,<, bits, fields, sign, ms)			\
BINOP(le##bits##uors##mors,<=, bits, fields, sign, ms)			\
BINOP(gt##bits##uors##mors,>, bits, fields, sign, ms)			\
BINOP(ge##bits##uors##mors,>=, bits, fields, sign, ms)


/* Macro instances to generate the actual function definitions described
   in the above macros.
*/

DECLARE(1,64, u, un, m, modular)
DECLARE(1,64, s, , m, modular)
DECLARE(1,64, u, un, s, saturation)
DECLARE(1,64, s, , s, saturation)

DECLARE(2,32, u, un, m, modular)
DECLARE(2,32, s, , m, modular)
DECLARE(2,32, u, un, s, saturation)
DECLARE(2,32, s, , s, saturation)

DECLARE(4,16, u, un, m, modular)
DECLARE(4,16, s, , m, modular)
DECLARE(4,16, u, un, s, saturation)
DECLARE(4,16, s, , s, saturation)

DECLARE(8,8, u, un, m, modular)
DECLARE(8,8, s, , m, modular)
DECLARE(8,8, u, un, s, saturation)
DECLARE(8,8, s, , s, saturation)

DECLARE(16,4, u, un, m, modular)
DECLARE(16,4, s, , m, modular)
DECLARE(16,4, u, un, s, saturation)
DECLARE(16,4, s, , s, saturation)

DECLARE(32,2, u, un, m, modular)
DECLARE(32,2, s, , m, modular)
DECLARE(32,2, u, un, s, saturation)
DECLARE(32,2, s, , s, saturation)



/************** MAIN PROGRAM *************************************************/

/* Test loop for operations with C equivalents */
#define TESTLOOP1(name, op, bits, uors, sign, letter, mors, ms)		\
if (!strcmp(#uors, "s"))						\
{									\
	k = i & (unsigned int)((1ULL<<(unsigned long long)bits)-1ULL);	\
	l = j & (unsigned int)((1ULL<<(unsigned long long)bits)-1ULL);	\
	i = sign_extend (i, bits);					\
	j = sign_extend (j, bits);					\
}									\
									\
/* Call SWARC function */						\
name##bits##uors##mors(&i, &j, &rval);					\
									\
/* Calculate expected value */						\
single = (sign##signed long long) (i op j);				\
									\
/* Convert conditionals to field mask */				\
if ( (!strcmp(#op, "==")) ||						\
    (!strcmp(#op, "!=")) ||						\
    (!strcmp(#op, "<")) ||						\
    (!strcmp(#op, "<=")) ||						\
    (!strcmp(#op, ">")) ||						\
    (!strcmp(#op, ">=")) ||						\
    (!strcmp(#op, "&&")) ||						\
    (!strcmp(#op, "||")) )						\
	single = (single)? 0xffffffffffffffffULL: 0ULL;			\
									\
/* Saturate if necessary */						\
if (strcmp(#ms, "saturation")) {					\
	single = single & ((1ULL<<bits) - 1ULL);			\
} else {								\
	if (strcmp(#uors, "s")) { /* Unsigned sat */			      \
		if (single > ((1ULL<<bits) - 1ULL))			      \
			single = ((1ULL<<bits) - 1ULL);			      \
		if ( (!strcmp(#op, "-")) && (i < j) )			      \
			single = 0ULL;					      \
	} else { /* Signed sat */					      \
		if (single > (signed long long)((1ULL<<(bits-1)) - 1ULL))     \
			single = ((1ULL<<(bits-1)) - 1ULL);		      \
		if (single < (signed long long)(~((1ULL<<(bits-1)) - 1ULL)))  \
			single = (1ULL<<(bits-1));			      \
		single &= ((1ULL<<bits) - 1ULL);			      \
	}								      \
}									\
									\
/* Convert single field expected value to replicated fragment */	\
actual = 0ULL;								\
for (count=0; count<(64ULL/bits); ++count)				\
{									\
	actual = actual | (single << (count*bits));			\
}									\
									\
/* Output message if Scc value does not match expected */		\
if (actual != rval.letter##q)						\
{									\
	if (strcmp(#op, "%"))						\
	{								\
		if (!strcmp(#uors, "s"))				\
			printf ( "0x%x" #op "0x%x yielded ", k, l );	\
		else							\
			printf ( "0x%x" #op "0x%x yielded ", i, j );	\
	}								\
	else								\
	{								\
		if (!strcmp(#uors, "s"))				\
			printf ( "0x%x%%0x%x yielded ", k, l );		\
		else							\
			printf ( "0x%x%%0x%x yielded ", i, j );		\
	}								\
	printf ( "0x%016llx.  Was expecting 0x%016llx.\n",		\
		 rval.letter##q, actual );				\
}



/* Test loop for operations that must be emulated in C */
#define TESTLOOP2(name, op, bits, uors, sign, letter, mors, ms)		\
if (!strcmp(#uors, "s"))						\
{									\
	k = i & (unsigned int)((1ULL<<(unsigned long long)bits)-1ULL);	\
	l = j & (unsigned int)((1ULL<<(unsigned long long)bits)-1ULL);	\
	i = sign_extend (i, bits);					\
	j = sign_extend (j, bits);					\
}									\
									\
/* Call SWARC function */						\
name##bits##uors##mors(&i, &j, &rval);					\
									\
/* Calculate expected value */						\
if (!strcmp(#op, "?<"))							\
	single =							\
	  ((sign##signed long long)((i<j)?i:j)) & ((1ULL<<bits)-1ULL);	\
else if (!strcmp(#op, "?>"))						\
	single =							\
	  ((sign##signed long long)((i>j)?i:j)) & ((1ULL<<bits)-1ULL);	\
else if (!strcmp(#op, "+/"))						\
{									\
	single = (sign##signed long long)(i+j);				\
	/* Rounds positively */						\
	if ((!strcmp(#uors, "u")) || (single>=0LL))			\
		single = (single+1)/2;					\
	else								\
		single = single/2;					\
	single &= (1ULL<<bits)-1ULL;					\
}									\
/* Note: min, max, and avg do not change with saturation */		\
									\
/* Convert single field expected value to replicated fragment */	\
actual = 0ULL;								\
for (count=0; count<(64ULL/bits); ++count)				\
{									\
	actual = actual | (single << (count*bits));			\
}									\
									\
/* Output message if Scc value does not match expected */		\
if (actual != rval.letter##q)						\
{									\
	if (!strcmp(#uors, "s"))					\
		printf ( "0x%x" #op "0x%x yielded 0x%016llx.  "		\
			 "Was expecting 0x%016llx.\n",			\
			 k, l, rval.letter##q, actual );		\
	else								\
		printf ( "0x%x" #op "0x%x yielded 0x%016llx.  "		\
			 "Was expecting 0x%016llx.\n",			\
			 i, j, rval.letter##q, actual );		\
}



#define SUITE(bits, uors, sign, letter, mors, ms)			\
{									\
	sign##signed long long x, y;					\
	sign##signed int i, j;						\
	sign##signed int k, l;						\
	sign##signed long long count, actual, single;			\
	time_t timer;							\
									\
	rval.uq = 0ULL;							\
									\
	time(&timer);							\
	printf ( "\nStarting test suite for " #bits " bit%s %s " #ms ": %s",\
		 (strcmp(#bits,"1"))? "s" : "",				\
		 (#uors##[0] == 'u')? "unsigned" : "signed",		\
		 ctime(&timer));					\
	fflush ( stdout );						\
									     \
	for (x=0ULL; x<(1ULL<<bits); x=(bits!=32)? x+1ULL: x+inc1)	     \
	{								     \
		xval = (unsigned long long) x;				     \
		i = (sign##signed int)(x & 0xffffffffULL);		     \
		for (y=0ULL; y<(1ULL<<bits); y=(bits!=32)? y+1ULL: y+inc2)   \
		{							     \
			yval = (unsigned long long) y;			     \
			j = (sign##signed int)(y & 0xffffffffULL);	     \
			TESTLOOP1(add, +, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(sub, -, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(mul, *, bits, uors, sign, letter, mors, ms)\
		/* Skip these until support for variable count shifts is added
			TESTLOOP1(shl, <<, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(shr, >>, bits, uors, sign, letter, mors, ms)\
		*/							\
			TESTLOOP2(min, ?<, bits, uors, sign, letter, mors, ms)\
			TESTLOOP2(max, ?>, bits, uors, sign, letter, mors, ms)\
			TESTLOOP2(avg, +/, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(eq, ==, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(ne, !=, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(lt, <, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(le, <=, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(gt, >, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(ge, >=, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(land, &&, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(lor, ||, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(and, &, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(andn, &~, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(or, |, bits, uors, sign, letter, mors, ms)\
			TESTLOOP1(xor, ^, bits, uors, sign, letter, mors, ms)\
			if ( j != 0 )					\
			{						\
				TESTLOOP1(div,/,bits,uors,sign,letter,mors,ms)\
				TESTLOOP1(mod,%,bits,uors,sign,letter,mors,ms)\
			}						\
		}							\
	}								\
	time(&timer);							\
	printf ( "\nFinished at %s", ctime(&timer));			\
	fflush ( stdout );						\
}



${
$include <stdio.h>
$include <string.h>
$include <time.h>
$include <signal.h>

/* Globals to make operands' values available to the signal handler */
unsigned long long xval, yval;

signed int sign_extend(signed int i, int bits)
{
	signed int shift = 32-bits;
	return ((i << shift) >> shift);
}

void newSIGHUP(int junk)
{
	fprintf(stderr, "\nvalid:xval=0x%016llx, yval=0x%016llx\n", xval, yval);
	signal(SIGHUP, newSIGHUP);
}

int main (int argc, char *argv[])
{
	p64_t rval;
	void *oldSIGHUP;
	unsigned long long inc1=1ULL;
	unsigned long long inc2=1ULL;

	if (argc == 1)
		fprintf(stderr, "Note usage: valid64 [16um] [16sm] [16us] "
				"[16ss] [32um [inc1 inc2]] [32sm [inc1 inc2]] "
				"[32us [inc1 inc2]] [32ss [inc1 inc2]]\n" );

	oldSIGHUP = signal(SIGHUP, newSIGHUP);

	if (argc == 1) {
		SUITE(1,u,un,u,m,modular)
		SUITE(1,s,,,m,modular)
		SUITE(1,u,un,u,s,saturation)
		SUITE(1,s,,,s,saturation)

		SUITE(2,u,un,u,m,modular)
		SUITE(2,s,,,m,modular)
		SUITE(2,u,un,u,s,saturation)
		SUITE(2,s,,,s,saturation)

		SUITE(4,u,un,u,m,modular)
		SUITE(4,s,,,m,modular)
		SUITE(4,u,un,u,s,saturation)
		SUITE(4,s,,,s,saturation)

		SUITE(8,u,un,u,m,modular)
		SUITE(8,s,,,m,modular)
		SUITE(8,u,un,u,s,saturation)
		SUITE(8,s,,,s,saturation)
	}

	if ((argv[1]) && (!strcmp(argv[1], "16um"))) {
		++argv;
		--argc;
		SUITE(16,u,un,u,m,modular)
	}

	if ((argv[1]) && (!strcmp(argv[1], "16sm"))) {
		++argv;
		--argc;
		SUITE(16,s,,,m,modular)
	}

	if ((argv[1]) && (!strcmp(argv[1], "16us"))) {
		++argv;
		--argc;
		SUITE(16,u,un,u,s,saturation)
	}

	if ((argv[1]) && (!strcmp(argv[1], "16ss"))) {
		++argv;
		--argc;
		SUITE(16,s,,,s,saturation)
	}


	if ((argv[1]) && (!strcmp(argv[1], "32um"))) {
		++argv;
		--argc;
		if (argv[1]) {
			if ((strcmp(argv[1], "32sm"))
			 && (strcmp(argv[1], "32us"))
			 && (strcmp(argv[1], "32ss"))) {
				inc1 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		if (argv[1]) {
			if ((strcmp(argv[1], "32sm"))
			 && (strcmp(argv[1], "32us"))
			 && (strcmp(argv[1], "32ss"))) {
				inc2 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		SUITE(32,u,un,u,m,modular)
	}

	if ((argv[1]) && (!strcmp(argv[1], "32sm"))) {
		++argv;
		--argc;
		if (argv[1]) {
			if ((strcmp(argv[1], "32us"))
			 && (strcmp(argv[1], "32ss"))) {
				inc1 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		if (argv[1]) {
			if ((strcmp(argv[1], "32us"))
			 && (strcmp(argv[1], "32ss"))) {
				inc2 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		SUITE(32,s,,,m,modular)
	}

	if ((argv[1]) && (!strcmp(argv[1], "32us"))) {
		++argv;
		--argc;
		if (argv[1]) {
			if (strcmp(argv[1], "32ss")) {
				inc1 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		if (argv[1]) {
			if (strcmp(argv[1], "32ss")) {
				inc2 = atoi(argv[1]);
				++argv;
				--argc;
			}
		}
		SUITE(32,u,un,u,s,saturation)
	}

	if ((argv[1]) && (!strcmp(argv[1], "32ss"))) {
		++argv;
		--argc;
		if (argv[1]) {
			inc1 = atoi(argv[1]);
			++argv;
			--argc;
		}
		if (argv[1]) {
			inc2 = atoi(argv[1]);
			++argv;
			--argc;
		}

		SUITE(32,s,,,s,saturation)
	}


	signal(SIGHUP, oldSIGHUP);

	return 0;
}
$}

