/* stdswar.c ******************************************************************
 Coding History:
	06-05-99 R.Fisher	Started strto128u() and strto64u().
	06-06-99 R.Fisher	Continued strto128u() and strto64u().
	06-07-99 R.Fisher	Split from showpart.c.
	06-08-99 R.Fisher	Added some cvt* functions.
	07-06-99 R.Fisher	No longer expects swartypes.h to be installed.
				Corrected leading whitespace strip code to not
				  loop forever in strto128u() and strto64u().
				Changed initialization of rval* from 0LL to 0ULL
				  in strto128u() and strto64u().
				Changed initialization of rval to rval.uq in
				  strto64u().
******************************************************************************/

#include <ctype.h>
#include "../swartypes.h"

int isbasedigit(int c, int base)
{
	if ((c >= '0') && (c <= '9'))		/* Numeral */
		return ( ((c-'0') < base)? 1:0 );

	if ((c >= 'A') && (c <= 'Z'))		/* Upr case letter */
		return ( ((c-'A') < base)? 1:0 );

	if ((c >= 'a') && (c <= 'z'))		/* Lwr case letter */
		return ( ((c-'a') < base)? 1:0 );

	return 0;
}

p128_t strto128u(const char *nptr, char **endptr, int base)
{
	char *ptr = (char *) nptr;
	p128_t rval;
	p128_t p1;
	p128_t tmp;

	rval.uq[1] = rval.uq[0] = p1.uq[1] = p1.uq[0] = 0ULL;

	/* Man page for strtoul() doesn't specify what to do for these,
	   but strtoul() seems to use base 10. */
	if ((base<0) || (base==1) || (base>36)) base = 10;

	/* Allow an arbitrary amount of whitespace as defined by isspace(3) */
	while ( isspace(*ptr) ) ++ptr;

	/* Base 16 is allowed to have a leading "0x" */
	if ((base==16) && (*ptr=='0') && ((ptr[1]=='x') || (ptr[1]=='X')))
	{
		ptr++;
		ptr++;
	}

	/* Base 0 can be base 16 (0x...), base 8 (0...), or base 10 (...) */
	if (base==0)
	{
		if (*ptr=='0')
		{
			if ((ptr[1]=='x') || (ptr[1]=='X'))
			{
				/* Treat as base 16 */
				base=16;
				ptr++;
				ptr++;
			}
			else
			{
				/* Treat as base 8 */
				base=8;
				ptr++;
			}
		}
		else
		{
			/* Treat as base 10 */
			base=10;
		}
	}

	/* Optional sign is ignored */
	if ((*ptr=='+') || (*ptr=='-')) ptr++;

	/* Handle characters which are valid digits for the current base */
	while ( isbasedigit(*ptr, base) )
	{
		int i;

		/* Conceptually, do: rval *= (unsigned long long) base; */
		/* For top half, do shift-add sequence.  For example:
		   10x = 8x + 2x = x<<3 + x<<1
		*/

		/* Accumulate into p1 */
		p1.uq[1] = 0LL;
		p1.uq[0] = 0LL;
		for (i=0; i<6; ++i)
		{
		    if ( base & (1<<i) )
		    {
			p1.uq[1] += (rval.uq[1] << i) | (rval.uq[0] >> (64-i));
		    }
		}
		p1.uq[0] = (unsigned long long) base * rval.uq[0];
		rval.uq[1] = p1.uq[1];
		rval.uq[0] = p1.uq[0];

		/* Because we know it is a valid digit, we only need to check
		   for what range it is in (i.e. lower bound), not ensure that
		   it is below the upper bound.
		*/
		/* Save for overflow check */
		tmp.uq[0] = rval.uq[0];

		/* Add in next digit */
		if (*ptr >= 'a') {
			rval.uq[0] += (*ptr - 87U);
		} else if (*ptr >= 'A') {
			rval.uq[0] += (*ptr - 55U);
		} else {
			rval.uq[0] += (*ptr - 0x30U);
		}

		/* Check for overflow.  If so, carry into upper half */
		if (tmp.uq[0] > rval.uq[0])
			rval.uq[1]++;

		ptr++;
	}

	/* If non-NULL, set *endptr to address of the first invalid char */
	if ( endptr ) *endptr = ptr;

	return rval;
}

#define strtoull(nptr, endptr, base)	(strto64u(nptr, endptr, base).uq)
p64_t strto64u(const char *nptr, char **endptr, int base)
{
	char *ptr = (char *) nptr;
	p64_t rval;

	rval.uq = 0ULL;

	/* Man page for strtoul() doesn't specify what to do for these,
	   but strtoul() seems to use base 10. */
	if ((base<0) || (base==1) || (base>36)) base = 10;

	/* Allow an arbitrary amount of whitespace as defined by isspace(3) */
	while ( isspace(*ptr) ) ++ptr;

	/* Base 16 is allowed to have a leading "0x" */
	if ((base==16) && (*ptr=='0') && ((ptr[1]=='x') || (ptr[1]=='X')))
	{
		ptr++;
		ptr++;
	}

	/* Base 0 can be base 16 (0x...), base 8 (0...), or base 10 (...) */
	if (base==0)
	{
		if (*ptr=='0')
		{
			if ((ptr[1]=='x') || (ptr[1]=='X'))
			{
				/* Treat as base 16 */
				base=16;
				ptr++;
				ptr++;
			}
			else
			{
				/* Treat as base 8 */
				base=8;
				ptr++;
			}
		}
		else
		{
			/* Treat as base 10 */
			base=10;
		}
	}

	/* Optional sign is ignored */
	if ((*ptr=='+') || (*ptr=='-')) ptr++;

	/* Handle characters which are valid digits for the current base */
	while ( isbasedigit(*ptr, base) )
	{
		rval.uq *= (unsigned long long) base;

		/* Because we know it is a valid digit, we only need to check
		   for what range it is in (i.e. lower bound), not ensure that
		   it is below the upper bound.
		*/
		if (*ptr >= 'a')
			rval.uq += (*ptr - 87U);
		else if (*ptr >= 'A')
			rval.uq += (*ptr - 55U);
		else 
			rval.uq += (*ptr - 0x30U);
		ptr++;
	}

	/* If non-NULL, set *endptr to address of the first invalid char */
	if ( endptr ) *endptr = ptr;

	return rval;
}


p128_t cvt1x4uto1x128u(unsigned char src)
{
	p128_t rval;
     
	rval.uq[1] = rval.uq[0] = 0ULL;
	rval.ub[0] = src & 0xfU;
	return rval;
}

p128_t cvt1x8uto1x128u(unsigned char src)
{
	p128_t rval;
     
	rval.uq[1] = rval.uq[0] = 0ULL;
	rval.ub[0] = src;
	return rval;
}

p128_t cvt1x16uto1x128u(unsigned int src)
{
	p128_t rval;
     
	rval.uq[1] = rval.uq[0] = 0ULL;
	rval.uw[0] = src;
	return rval;
}

p128_t cvt1x32uto1x128u(p32_t src)
{
	p128_t rval;
     
	rval.uq[1] = rval.uq[0] = 0ULL;
	rval.ud[0] = src.ud;
	return rval;
}

p128_t cvt1x64uto1x128u(p64_t src)
{
	p128_t rval;
     
	rval.uq[1] = 0ULL;
	rval.uq[0] = src.uq;
	return rval;
}


/* Replicates the low 4 bits of src, ignoring the upper 4 bits. */
p128_t cvt1x4uto32x4u(unsigned char src)
{
	p128_t rval;
	int i;

	for (i=0; i<16; ++i)
		rval.ub[i] = (src & 0xfU) | ((src & 0xfU) << 4);
	return rval;
}

p128_t cvt1x8uto16x8u(unsigned char src)
{
	p128_t rval;
	int i;

	for (i=0; i<16; ++i)
		rval.ub[i] = src & 0xffU;
	return rval;
}

p128_t cvt1x16uto8x16u(unsigned int src)
{
	p128_t rval;
	int i;

	for (i=0; i<8; ++i)
		rval.uw[i] = src & 0xffffU;
	return rval;
}

p128_t cvt1x32uto4x32u(p32_t src)
{
	p128_t rval;
	int i;

	for (i=0; i<4; ++i)
		rval.ud[i] = src.ud;
	return rval;
}

p128_t cvt1x64uto2x64u(p64_t src)
{
	p128_t rval;
	int i;

	for (i=0; i<2; ++i)
		rval.uq[i] = src.uq;
	return rval;
}



p64_t cvt1x4uto1x64u(unsigned char src)
{
	p64_t rval;
     
	rval.uq = 0ULL;
	rval.ub[0] = src & 0xfU;
	return rval;
}

p64_t cvt1x8uto1x64u(unsigned char src)
{
	p64_t rval;
     
	rval.uq = 0ULL;
	rval.ub[0] = src;
	return rval;
}

p64_t cvt1x16uto1x64u(unsigned int src)
{
	p64_t rval;
     
	rval.uq = 0ULL;
	rval.uw[0] = src;
	return rval;
}

p64_t cvt1x32uto1x64u(p32_t src)
{
	p64_t rval;
     
	rval.uq = 0ULL;
	rval.ud[0] = src.ud;
	return rval;
}


/* Replicates the low 4 bits of src, ignoring the upper 4 bits. */
p64_t cvt1x4uto16x4u(unsigned char src)
{
	p64_t rval;
	int i;

	for (i=0; i<8; ++i)
		rval.ub[i] = (src & 0xfU) | ((src & 0xfU) << 4);
	return rval;
}

p64_t cvt1x8uto8x8u(unsigned char src)
{
	p64_t rval;
	int i;

	for (i=0; i<8; ++i)
		rval.ub[i] = src & 0xffU;
	return rval;
}

p64_t cvt1x16uto4x16u(unsigned int src)
{
	p64_t rval;
	int i;

	for (i=0; i<4; ++i)
		rval.uw[i] = src & 0xffffU;
	return rval;
}

p64_t cvt1x32uto2x32u(p32_t src)
{
	p64_t rval;
	int i;

	for (i=0; i<2; ++i)
		rval.ud[i] = src.ud;
	return rval;
}



p32_t cvt1x4uto1x32u(unsigned char src)
{
	p32_t rval;
     
	rval.ud = 0U;
	rval.ub[0] = src & 0xfU;
	return rval;
}

p32_t cvt1x8uto1x32u(unsigned char src)
{
	p32_t rval;
     
	rval.ud = 0U;
	rval.ub[0] = src;
	return rval;
}

p32_t cvt1x16uto1x32u(unsigned int src)
{
	p32_t rval;
     
	rval.ud = 0U;
	rval.uw[0] = src;
	return rval;
}


/* Replicates the low 4 bits of src, ignoring the upper 4 bits. */
p32_t cvt1x4uto8x4u(unsigned char src)
{
	p32_t rval;
	int i;

	for (i=0; i<4; ++i)
		rval.ub[i] = (src & 0xfU) | ((src & 0xfU) << 4);
	return rval;
}

p32_t cvt1x8uto4x8u(unsigned char src)
{
	p32_t rval;
	int i;

	for (i=0; i<4; ++i)
		rval.ub[i] = src & 0xffU;
	return rval;
}

p32_t cvt1x16uto2x16u(unsigned int src)
{
	p32_t rval;
	int i;

	for (i=0; i<2; ++i)
		rval.uw[i] = src & 0xffffU;
	return rval;
}

