/*	max.h

	Multimedia extensions GCC interface library for PA-RISC 2.0.

	To use this library, simply include this header file
	and compile with GCC.  You MUST have inlining enabled
	in order for max_ok() to work; this can be done by
	simply using -O on the GCC command line.

	Compiling with -DMAX_TRACE will cause detailed trace
	output to be sent to stderr for each max operation.
	This adds lots of code, and obviously slows execution to
	a crawl, but can be very useful for debugging.

	THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY
	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
	LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY
	AND FITNESS FOR ANY PARTICULAR PURPOSE.

	1998-99 by R. Fisher

 History:
	981108	R.Fisher	Original Coding

	* Unreleased (internal or interim) versions

 Notes:
	NxB notation is not necessarily correct, it should be parameterized.

 Acknowledgments:
*/

#ifndef _MAX_H
#define _MAX_H




/*	Function to test if multimedia instructions are supported...
*/
inline extern int
mm_support(void)
{
	/* Returns 1 if MAX instructions are supported.
	   0 if hardware does not support MAX.
	*/
	register int rval = 1;

	/* Return */
	return(rval);
}

/*	Function to test if max instructions are supported...  */
inline extern int
max_ok(void)
{
	/* Returns 1 if MAX instructions are supported, 0 otherwise */
	return ( mm_support() & 0x1 );
}


/*	Helper functions for the instruction macros that follow...
	(note that memory-to-register, m2r, instructions are nearly
	 as efficient as register-to-register, r2r, instructions;
	 however, memory-to-memory instructions are really simulated
	 as a convenience, and are only 1/3 as efficient)
*/
#ifdef	MAX_TRACE

/*	Include the stuff for printing a trace to stderr...
*/

#include <stdio.h>

#define	max_i2r(op, imm, reg) \
	{ \
		p64_t max_trace; \
		max_trace = (imm); \
		fprintf(stderr, #op "_i2r(" #imm "=0x%016llx, ", max_trace.q); \
		__asm__ __volatile__ ("movq %%" #reg ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #reg "=0x%016llx) => ", max_trace.q); \
		__asm__ __volatile__ (#op " %0, %%" #reg \
				      : /* nothing */ \
				      : "X" (imm)); \
		__asm__ __volatile__ ("movq %%" #reg ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #reg "=0x%016llx\n", max_trace.q); \
	}

#define	max_m2r(op, mem, reg) \
	{ \
		p64_t max_trace; \
		max_trace = (mem); \
		fprintf(stderr, #op "_m2r(" #mem "=0x%016llx, ", max_trace.q); \
		__asm__ __volatile__ ("movq %%" #reg ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #reg "=0x%016llx) => ", max_trace.q); \
		__asm__ __volatile__ (#op " %0, %%" #reg \
				      : /* nothing */ \
				      : "X" (mem)); \
		__asm__ __volatile__ ("movq %%" #reg ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #reg "=0x%016llx\n", max_trace.q); \
	}

#define	max_r2m(op, reg, mem) \
	{ \
		p64_t max_trace; \
		__asm__ __volatile__ ("movq %%" #reg ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #op "_r2m(" #reg "=0x%016llx, ", max_trace.q); \
		max_trace = (mem); \
		fprintf(stderr, #mem "=0x%016llx) => ", max_trace.q); \
		__asm__ __volatile__ (#op " %%" #reg ", %0" \
				      : "=X" (mem) \
				      : /* nothing */ ); \
		max_trace = (mem); \
		fprintf(stderr, #mem "=0x%016llx\n", max_trace.q); \
	}

#define	max_r2r(op, regs, regd) \
	{ \
		p64_t max_trace; \
		__asm__ __volatile__ ("movq %%" #regs ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #op "_r2r(" #regs "=0x%016llx, ", max_trace.q); \
		__asm__ __volatile__ ("movq %%" #regd ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #regd "=0x%016llx) => ", max_trace.q); \
		__asm__ __volatile__ (#op " %" #regs ", %" #regd); \
		__asm__ __volatile__ ("movq %%" #regd ", %0" \
				      : "=X" (max_trace) \
				      : /* nothing */ ); \
		fprintf(stderr, #regd "=0x%016llx\n", max_trace.q); \
	}


#else

/*	These macros are a lot simpler without the tracing...
*/

#define	max_i2r(op, imm, reg) \
	__asm__ __volatile__ (#op " %0, " #imm ", %0" \
			      : "=r" (reg) \
			      : "r" (reg) )
#define	max_m2r(op, mem, reg) \
	__asm__ __volatile__ (#op " %0, %%" #reg \
			      : /* nothing */ \
			      : "X" (mem))
#define	max_r2m(op, reg, mem) \
	__asm__ __volatile__ (#op " %%" #reg ", %0" \
			      : "=X" (mem) \
			      : /* nothing */ )
#define	max_r2r(op, regs, regd) \
	__asm__ __volatile__ (#op " %%" #regs ", %%" #regd)
/*
#define	max_i2r_c(op, cmplt, imm, reg) \
	__asm__ __volatile__ (#op "," #cmplt " %0, " #imm ", %0" \
			      : "=r" (reg) \
			      : "r" (reg))
*/
#define	max_r2r_i(op, imm, reg) \
	__asm__ __volatile__ (#op "," #imm " %0, %0" \
			      : "=r" (reg) \
			      : "r" (reg))
#define	max_m2r_c(op, cmplt, mem, reg) \
	__asm__ __volatile__ (#op "," #cmplt " %0, %%" #reg \
			      : /* nothing */ \
			      : "X" (mem))
/*
#define	max_r2m_c(op, cmplt, reg, mem) \
	__asm__ __volatile__ (#op "," #cmplt " %%" #reg ", %0" \
			      : "=X" (mem) \
			      : /* nothing */ )
*/
#define	max_r2r_c(op, cmplt, regs, regd) \
	__asm__ __volatile__ (#op "," #cmplt " %%" #regs ", %%" #regd)
#endif


/*	1x64 MOVe Quadword
	(this is both a load and a store...
	 in fact, it is the only way to store)
*/
#define	movq_m2r(var, reg)	max_m2r(movq, var, reg)
#define	movq_r2m(reg, var)	max_r2m(movq, reg, var)
#define	movq_r2r(regs, regd)	max_r2r(movq, regs, regd)
#define	movq(vars, vard) \
	__asm__ __volatile__ ("movq %1, %%r0\n\t" \
			      "movq %%r0, %0" \
			      : "=X" (vard) \
			      : "X" (vars))


/*	1x32 MOVe Doubleword
	(like movq, this is both load and store...
	 but is most useful for moving things between
	 max registers and ordinary registers)
*/
#define	movd_m2r(var, reg)	max_m2r(movd, var, reg)
#define	movd_r2m(reg, var)	max_r2m(movd, reg, var)
#define	movd_r2r(regs, regd)	max_r2r(movd, regs, regd)
#define	movd(vars, vard) \
	__asm__ __volatile__ ("movd %1, %%r0\n\t" \
			      "movd %%r0, %0" \
			      : "=X" (vard) \
			      : "X" (vars))



/*	1x32 Word ADD (p7-57)
*/
#define	add_m2r(var, reg)	max_m2r(add, var, reg)
#define	add_r2r(regs, regd)	max_r2r(add, regs, regd)


/*	2x16 Half-word parallel ADDs (p7-57)
*/
#define	hadd_m2r(var, reg)	max_m2r(hadd, var, reg)
#define	hadd_r2r(regs, regd)	max_r2r(hadd, regs, regd)


/*	2x16 Half-word parallel ADDs using Saturation arithmetic (p7-57)
*/
#define	haddss_m2r(var, reg)	max_m2r_c(hadd, ss, var, reg)
#define	haddss_r2r(regs, regd)	max_r2r_c(hadd, ss, regs, regd)


/*	2x16 Half-word parallel ADDs using Unsigned Saturation arithmetic (p7-57)
*/
#define	haddus_m2r(var, reg)	max_m2r_c(hadd, us, var, reg)
#define	haddus_r2r(regs, regd)	max_r2r_c(hadd, us, regs, regd)



/*	1x32 Word SUB (p7-63)
*/
#define	sub_m2r(var, reg)	max_m2r(sub, var, reg)
#define	sub_r2r(regs, regd)	max_r2r(sub, regs, regd)


/*	2x16 Half-word parallel SUBs (p7-63)
*/
#define	hsub_m2r(var, reg)	max_m2r(hsub, var, reg)
#define	hsub_r2r(regs, regd)	max_r2r(hsub, regs, regd)


/*	2x16 Half-word parallel SUBs using Saturation arithmetic (p7-63)
*/
#define	hsubss_m2r(var, reg)	max_m2r_c(hsub, ss, var, reg)
#define	hsubss_r2r(regs, regd)	max_r2r_c(hsub, ss, regs, regd)


/*	2x16 Half-word parallel SUBs using Unsigned Saturation arithmetic (p7-63)
*/
#define	hsubus_m2r(var, reg)	max_m2r_c(hsub, us, var, reg)
#define	hsubus_r2r(regs, regd)	max_r2r_c(hsub, us, regs, regd)



/*	2x16 Half-word parallel AVeraGes (p7-58)
*/
#define	havg_m2r(var, reg)	max_m2r(havg, var, reg)
#define	havg_r2r(regs, regd)	max_r2r(havg, regs, regd)


/*	2x16 Half-word parallel SHift Left (p7-59)
*/
#define	hshl_i2r(imm, reg)	max_i2r(hshl, imm, reg)
#define	hshl_m2r(var, reg)	max_m2r(hshl, var, reg)
#define	hshl_r2r(regs, regd)	max_r2r(hshl, regs, regd)


/*	2x16 Half-word parallel SHift Left and parallel ADD (p7-60)
*/
#define	hshladd_i2r(imm, reg)	max_i2r(hshladd, imm, reg)
#define	hshladd_m2r(var, reg)	max_m2r(hshladd, var, reg)
#define	hshladd_r2r(regs, regd)	max_r2r(hshladd, regs, regd)


/*	2x16 Half-word parallel SHift Right (p7-61)
*/
#define	hshr_i2r(imm, reg)	max_i2r(hshr, imm, reg)
#define	hshr_m2r(var, reg)	max_m2r(hshr, var, reg)
#define	hshr_r2r(regs, regd)	max_r2r(hshr, regs, regd)


/*	2x16 Half-word parallel SHift Right and parallel ADD (p7-62)
*/
#define	hshradd_i2r(imm, reg)	max_i2r(hshradd, imm, reg)
#define	hshradd_m2r(var, reg)	max_m2r(hshradd, var, reg)
#define	hshradd_r2r(regs, regd)	max_r2r(hshradd, regs, regd)


/*	1x64, 2x32, and 4x16 Parallel Shift Left Logical
#define	psllq_i2r(imm, reg)	max_m2r(psllq, imm, reg)
#define	psllq_m2r(var, reg)	max_m2r(psllq, var, reg)
#define	psllq_r2r(regs, regd)	max_r2r(psllq, regs, regd)
*/


/*      1x64 bitwise AND
*/
#define and_m2r(var, reg)      max_m2r(and, var, reg)
#define and_r2r(regs, regd)    max_r2r(and, regs, regd)


/*      1x64 bitwise AND with Not the destination
*/
#define andn_m2r(var, reg)     max_m2r(andc, var, reg)
#define andn_r2r(regs, regd)   max_r2r(andc, regs, regd)


/*      1x64 bitwise OR
*/
#define or_m2r(var, reg)       max_m2r(or, var, reg)
#define or_r2r(regs, regd)     max_r2r(or, regs, regd)


/*      1x64 bitwise eXclusive OR
*/
#define xor_m2r(var, reg)      max_m2r(xor, var, reg)
#define xor_r2r(regs, regd)    max_r2r(xor, regs, regd)


/*	2x16 MIX Half-words Left (p7-92)
*/
#define	mixhl_m2r(var, reg)	max_m2r_c(mixh, l, var, reg)
#define	mixhl_r2r(regs, regd)	max_r2r_c(mixh, l, regs, regd)


/*	2x16 MIX Half-words Right (p7-92)
*/
#define	mixhr_m2r(var, reg)	max_m2r_c(mixh, r, var, reg)
#define	mixhr_r2r(regs, regd)	max_r2r_c(mixh, r, regs, regd)


/*	2x16 MIX Words Left (p7-93)
*/
#define	mixwl_m2r(var, reg)	max_m2r_c(mixw, l, var, reg)
#define	mixwl_r2r(regs, regd)	max_r2r_c(mixw, l, regs, regd)


/*	2x16 MIX Words Right (p7-93)
*/
#define	mixwr_m2r(var, reg)	max_m2r_c(mixw, r, var, reg)
#define	mixwr_r2r(regs, regd)	max_r2r_c(mixw, r, regs, regd)


/*	2x16 PERMute Half-words (p7-106)
*/
#define	permh_i2r(imm, reg)	max_r2r_i(permh, imm, reg)
#define	permh_m2r(var, reg)	max_m2r(permh, var, reg)
#define	permh_r2r(regs, regd)	max_r2r(permh, regs, regd)


/*	1x64 CoMPare and CLeaR (p7-27)
*/
#define	cmpclr(cond, regs, regd) max_r2r(cmpclr##cond, regs, regd)


/*	derived functions...
*/

#define	pdivub_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.b[0]) _reg.ub[0] /= _mem.ub[0]; \
		if (_mem.b[1]) _reg.ub[1] /= _mem.ub[1]; \
		if (_mem.b[2]) _reg.ub[2] /= _mem.ub[2]; \
		if (_mem.b[3]) _reg.ub[3] /= _mem.ub[3]; \
		if (_mem.b[4]) _reg.ub[4] /= _mem.ub[4]; \
		if (_mem.b[5]) _reg.ub[5] /= _mem.ub[5]; \
		if (_mem.b[6]) _reg.ub[6] /= _mem.ub[6]; \
		if (_mem.b[7]) _reg.ub[7] /= _mem.ub[7]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivub_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivub_m2r(_regs, regd); \
	}

#define	pdivb_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.b[0]) _reg.b[0] /= _mem.b[0]; \
		if (_mem.b[1]) _reg.b[1] /= _mem.b[1]; \
		if (_mem.b[2]) _reg.b[2] /= _mem.b[2]; \
		if (_mem.b[3]) _reg.b[3] /= _mem.b[3]; \
		if (_mem.b[4]) _reg.b[4] /= _mem.b[4]; \
		if (_mem.b[5]) _reg.b[5] /= _mem.b[5]; \
		if (_mem.b[6]) _reg.b[6] /= _mem.b[6]; \
		if (_mem.b[7]) _reg.b[7] /= _mem.b[7]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivb_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivb_m2r(_regs, regd); \
	}

#define	pdivuw_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.w[0]) _reg.uw[0] /= _mem.uw[0]; \
		if (_mem.w[1]) _reg.uw[1] /= _mem.uw[1]; \
		if (_mem.w[2]) _reg.uw[2] /= _mem.uw[2]; \
		if (_mem.w[3]) _reg.uw[3] /= _mem.uw[3]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivuw_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivuw_m2r(_regs, regd); \
	}

#define	pdivw_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.w[0]) _reg.w[0] /= _mem.w[0]; \
		if (_mem.w[1]) _reg.w[1] /= _mem.w[1]; \
		if (_mem.w[2]) _reg.w[2] /= _mem.w[2]; \
		if (_mem.w[3]) _reg.w[3] /= _mem.w[3]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivw_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivw_m2r(_regs, regd); \
	}

#define	pdivud_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.d[0]) _reg.ud[0] /= _mem.ud[0]; \
		if (_mem.d[1]) _reg.ud[1] /= _mem.ud[1]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivud_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivud_m2r(_regs, regd); \
	}

#define	pdivd_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.d[0]) _reg.d[0] /= _mem.d[0]; \
		if (_mem.d[1]) _reg.d[1] /= _mem.d[1]; \
		movq_m2r(_reg, reg); \
	}

#define	pdivd_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivd_m2r(_regs, regd); \
	}

#define	pdivuq_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.q) _reg.uq /= _mem.uq; \
		movq_m2r(_reg, reg); \
	}

#define	pdivuq_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivuq_m2r(_regs, regd); \
	}

#define	pdivq_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.q) _reg.q /= _mem.q; \
		movq_m2r(_reg, reg); \
	}

#define	pdivq_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pdivq_m2r(_regs, regd); \
	}


#define	pmodub_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.b[0]) _reg.ub[0] %= _mem.ub[0]; \
		if (_mem.b[1]) _reg.ub[1] %= _mem.ub[1]; \
		if (_mem.b[2]) _reg.ub[2] %= _mem.ub[2]; \
		if (_mem.b[3]) _reg.ub[3] %= _mem.ub[3]; \
		if (_mem.b[4]) _reg.ub[4] %= _mem.ub[4]; \
		if (_mem.b[5]) _reg.ub[5] %= _mem.ub[5]; \
		if (_mem.b[6]) _reg.ub[6] %= _mem.ub[6]; \
		if (_mem.b[7]) _reg.ub[7] %= _mem.ub[7]; \
		movq_m2r(_reg, reg); \
	}

#define	pmodub_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmodub_m2r(_regs, regd); \
	}

#define	pmodb_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.b[0]) _reg.b[0] %= _mem.b[0]; \
		if (_mem.b[1]) _reg.b[1] %= _mem.b[1]; \
		if (_mem.b[2]) _reg.b[2] %= _mem.b[2]; \
		if (_mem.b[3]) _reg.b[3] %= _mem.b[3]; \
		if (_mem.b[4]) _reg.b[4] %= _mem.b[4]; \
		if (_mem.b[5]) _reg.b[5] %= _mem.b[5]; \
		if (_mem.b[6]) _reg.b[6] %= _mem.b[6]; \
		if (_mem.b[7]) _reg.b[7] %= _mem.b[7]; \
		movq_m2r(_reg, reg); \
	}

#define	pmodb_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmodb_m2r(_regs, regd); \
	}

#define	pmoduw_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.w[0]) _reg.uw[0] %= _mem.uw[0]; \
		if (_mem.w[1]) _reg.uw[1] %= _mem.uw[1]; \
		if (_mem.w[2]) _reg.uw[2] %= _mem.uw[2]; \
		if (_mem.w[3]) _reg.uw[3] %= _mem.uw[3]; \
		movq_m2r(_reg, reg); \
	}

#define	pmoduw_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmoduw_m2r(_regs, regd); \
	}

#define	pmodw_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.w[0]) _reg.w[0] %= _mem.w[0]; \
		if (_mem.w[1]) _reg.w[1] %= _mem.w[1]; \
		if (_mem.w[2]) _reg.w[2] %= _mem.w[2]; \
		if (_mem.w[3]) _reg.w[3] %= _mem.w[3]; \
		movq_m2r(_reg, reg); \
	}

#define	pmodw_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmodw_m2r(_regs, regd); \
	}

#define	pmodud_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.d[0]) _reg.ud[0] %= _mem.ud[0]; \
		if (_mem.d[1]) _reg.ud[1] %= _mem.ud[1]; \
		movq_m2r(_reg, reg); \
	}

#define	pmodud_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmodud_m2r(_regs, regd); \
	}

#define	pmodd_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		if (_mem.d[0]) _reg.d[0] %= _mem.d[0]; \
		if (_mem.d[1]) _reg.d[1] %= _mem.d[1]; \
		movq_m2r(_reg, reg); \
	}

#define	pmodd_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmodd_m2r(_regs, regd); \
	}

#define	pmulld_m2r(mem, reg) \
	{ \
		p64_t _reg; \
		p64_t _mem = mem; \
		movq_r2m(reg, _reg); \
		_reg.d[0] *= _mem.d[0]; \
		_reg.d[1] *= _mem.d[1]; \
		movq_m2r(_reg, reg); \
	}

#define	pmulld_r2r(regs, regd) \
	{ \
		p64_t _regs; \
		movq_r2m(regs, _regs); \
		pmulld_m2r(_regs, regd); \
	}

#endif

