/*	coercer.c

	Coerce types to be correct within a tree
*/


#include "stdpccts.h"
#include "messages.h"


static int
diftypel(typ *t1,
typ *t2)
{
	/* Return 1 if types differ in saturation, unsignedness, base type
	   (char, short, int, long, long long, float, or swar type), or bits of
	   precision (field width).  Return 0 otherwise. */

	if ((t1->attr & (TYP_SAT |
			 TYP_UNSIGN |
			 TYP_CHAR |
			 TYP_SHORT |
			 TYP_INT |
			 TYP_LONG |
			 TYP_LLONG |
			 TYP_FLOAT |
			 TYP_SWAR)) !=
	    (t2->attr & (TYP_SAT |
			 TYP_UNSIGN |
			 TYP_CHAR |
			 TYP_SHORT |
			 TYP_INT |
			 TYP_LONG |
			 TYP_LLONG |
			 TYP_FLOAT |
			 TYP_SWAR))) {
		/* different type */
		return(1);
	}

	if (t1->bits != t2->bits) {
		/* different bit precision */
		return(1);
	}

	return(0);
}


inline int
diftypdim(typ *t1,
typ *t2)
{
	/* Return 1 if type dimension's differ, 0 otherwise. */
	return(t1->dim != t2->dim);
}


int
diftyp(typ t1,
typ t2)
{
	/* Return 1 if types differ in dimension, saturation, unsignedness,
	   base type (char, short, int, float, or swar type), or bits of
	   precision (field width).  Return 0 otherwise. */

	return(diftypdim(&t1, &t2) || diftypel(&t1, &t2));
}


void
incompat(typ t1,
typ t2)
{
	/* Issue error message if neither type's dimension is 1,
	   and they are not equal to each other. */

 	if ((t1.dim > 1) &&
	    (t2.dim > 1) &&
	    (t1.dim != t2.dim)) {
		char buf[512];
		snprintf(buf,
			512,
			"incompatible dimensions [%d] vs [%d]",
			t1.dim,
			t2.dim);
		error(buf);
	}
}


typ
retyp(typ t1,
typ t2)
{
	/* Determine, and return, the new type unifying these two types */

	/* Is either type implicit?  Return the other one if so */
	if (t1.dim == 0) return(t2);
	if (t2.dim == 0) return(t1);

	/* If one is a scalar, it gets promoted */
	if (t1.dim == 1) {
		t1.dim = t2.dim;
	} else if (t2.dim == 1) {
		t2.dim = t1.dim;
	}

	/* Do we have vectors of disparate lengths? */
	if (t1.dim != t2.dim) {
		/* Yup.  This should not happen, but if it does
		   we simply truncate the longer one
		*/
		if (t1.dim > t2.dim) {
			t1.dim = t2.dim;
		} else {
			t2.dim = t1.dim;
		}
	}

	/* SWAR types always beat C types for coercion,
	   even if bits per element gets reduced
	*/
	if ((t1.attr & TYP_SWAR) &&
	    (t2.attr & (TYP_CHAR|TYP_SHORT|TYP_INT|TYP_LONG|TYP_LLONG))) {
		t2.attr &= ~(TYP_CHAR|TYP_SHORT|TYP_INT|TYP_LONG|TYP_LLONG);
		t2.attr |= TYP_SWAR;
		t2.bits = t1.bits;
	} else if ((t2.attr & TYP_SWAR) &&
		   (t1.attr&(TYP_CHAR|TYP_SHORT|TYP_INT|TYP_LONG|TYP_LLONG))) {
		t1.attr &= ~(TYP_CHAR|TYP_SHORT|TYP_INT|TYP_LONG|TYP_LLONG);
		t1.attr |= TYP_SWAR;
		t1.bits = t2.bits;
	}

	/* Bigger element types win over smaller ones,
	   and we can copy C type from the bigger one
	*/
	if (t1.bits > t2.bits) {
		t2.bits = t1.bits;
		t2.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
			     TYP_LONG | TYP_LLONG | TYP_SWAR);
		t2.attr |= (t1.attr &
			    (TYP_CHAR | TYP_SHORT | TYP_INT |
			     TYP_LONG | TYP_LLONG | TYP_SWAR));
	} else if (t2.bits > t1.bits) {
		t1.bits = t2.bits;
		t1.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
			     TYP_LONG | TYP_LLONG | TYP_SWAR);
		t1.attr |= (t2.attr &
			    (TYP_CHAR | TYP_SHORT | TYP_INT |
			     TYP_LONG | TYP_LLONG | TYP_SWAR));
	}

	/* Float wins over integer */
	if ((t1.attr & TYP_FLOAT) ^ (t2.attr & TYP_FLOAT)) {
		/* Floats are 32 bit, not unsigned nor saturation */
		t1.bits = 32;
		t2.bits = 32;
		t1.attr |= TYP_FLOAT;
		t2.attr |= TYP_FLOAT;
		t1.attr &= ~(TYP_UNSIGN | TYP_SAT);
		t2.attr &= ~(TYP_UNSIGN | TYP_SAT);
	}

	/* Signed wins over unsigned */
	if ((t1.attr & TYP_UNSIGN) ^ (t2.attr & TYP_UNSIGN)) {
		t1.attr &= ~TYP_UNSIGN;
		t2.attr &= ~TYP_UNSIGN;
	}

	/* Saturation wins over modular */
	if ((t1.attr & TYP_SAT) ^ (t2.attr & TYP_SAT)) {
		t1.attr |= TYP_SAT;
		t2.attr |= TYP_SAT;
	}

	/* Now remove the meaningless attributes,
	   suggest that this should be in a register,
	   and return the unifying type
	*/
	t1.attr &= (TYP_SAT |
		    TYP_UNSIGN |
		    TYP_CHAR |
		    TYP_SHORT |
		    TYP_INT |
		    TYP_LONG |
		    TYP_LLONG |
		    TYP_SWAR |
		    TYP_FLOAT);
	t1.attr |= TYP_REG;
	return(t1);
}

