/*	cpool.c

	The constant pool utilities.

	Support for storing/accessing constant values during compilation,
	and generating code for an array holding the constant values that
	aren't eliminated via folding.

	Uses p128_ts internally, zeroing usused portions of the p128_t on
	entry and comparison based on register size of target.
*/

#undef DEBUG

#include "stdpccts.h"
#include "swartypes.h"
#include "tuple.h"
#include "oputils.h"
#include "cpool.h"
#include "output.h"
#include "messages.h"
#include "cpool.h"

static p128_t cpool[MAXCPOOL];

static int cpoolsp = 0;

static void
cpoolinit32(void)
{
	/* Declare the constant pool pointer */
	Ctab();

	fprintf(Cout, "extern p32_t cpool32[];\n");
	Ctab();
	fprintf(Cout, "register p32_t *_cpool = &(cpool32[0]);\n\n");
}

static void
cpoolinit64(void)
{
	/* Declare the constant pool pointer */
	Ctab();

	fprintf(Cout, "extern p64_t cpool64[];\n");
	Ctab();
	fprintf(Cout, "register p64_t *_cpool = &(cpool64[0]);\n\n");
}

static void
cpoolinit128(void)
{
	/* Declare the constant pool pointer */
	Ctab();

	fprintf(Cout, "extern p128_t cpool128[];\n");
	Ctab();
	fprintf(Cout, "register p128_t *_cpool = &(cpool128[0]);\n\n");
}

void
cpoolinit(void)
{
	/* Pre-enter the two most popular constants...
	   this ensures that the constant pool will exist,
	   and also that these two values will have the
	   shortest possible addressing form
	*/
	cpoolenter((p128_t){{0LL, 0LL}});
	cpoolenter((p128_t){{-1LL, -1LL}});

	/* Perform any target-specific initialization */
	switch (bitsperfrag()) {
	case 128:
		cpoolinit128();
		break;
	case 64:
		cpoolinit64();
		break;
	case 32:
		cpoolinit32();
		break;
	default:
		{
			char buf[50];
			snprintf(buf,
				 50,
				 "cpoolinit() doesn't handle %d-bit frags",
				 bitsperfrag());
			bug(buf);
		}
	}
}

void
cpoolenter(p128_t t)
{
	register int i;
	p128_t tmp = t;		/* Gcc 2.7.2.1 bombs when you assign t.q[1] */

	/* Zero parts of the value not used by the target */
	switch (bitsperfrag()) {
	case 32:
		tmp.ud[1] = 0U;
	case 64:
		tmp.uq[1] = 0ULL;
	case 128:
		break;
	default:
		{
			char buf[50];
			snprintf(buf,
				 50,
				 "cpoolenter() doesn't handle %d-bit frags",
				 bitsperfrag());
			warn(buf);
		}
	}

	/* If the value is already entered, just return */
	for (i=0; i<cpoolsp; ++i) {
		if ( (cpool[i].q[1] == tmp.q[1]) &&
		     (cpool[i].q[0] == tmp.q[0]) )
			return;
	}

	cpool[cpoolsp++] = tmp;
}

char *
cpoolname(p128_t t)
{
	/* Generate an access of a value in the constant pool */

	static char cpoolname[256];
	register int i;

	#ifdef DEBUG
		fprintf (Cout,
			 "/*\ncpoolname(): Trying to match %016llx,%016llx\n",
			 t.q[1],
			 t.q[0]);
	#endif
	for (i=0; i<cpoolsp; ++i) {
		#ifdef DEBUG
			fprintf (Cout,
				 "\t\t\t  to %016llx,%016llx\n",
				 cpool[i].q[1],
				 cpool[i].q[0]);
		#endif
		if ( (cpool[i].q[1] == t.q[1]) && (cpool[i].q[0] == t.q[0]) ) {
			snprintf(&(cpoolname[0]), 256, "*(_cpool + %d)", i);
			return(&(cpoolname[0]));
		}
	}
	#ifdef DEBUG
		fprintf (Cout, "*/\n");
	#endif

	/* I would normally return an emtpy string on error, but doing this
	   will cause a parse error in the output C code, so I might as well
	   have some useful information there.  Also, if I'm going to do this
	   I might as well do it every time this is called, so I'll do it here.
	*/
	snprintf(&(cpoolname[0]),
		 256,
		 "cpoolname() lookup of %016llx,%016llx failed",
		 t.q[1],
		 t.q[0]);
	return(&(cpoolname[0]));
}

static void
cpooldump32(void)
{
	if (cpoolsp) {
		register int i;

		fprintf(Cout, "\n/* SWAR Constant pool */\n");
		fprintf(Cout, "p32_t cpool32[] = {\n");
		/* fprintf(Cout, "unsigned cpool32[] = {\n"); */

		for (i=0; i<cpoolsp; ++i) {
			fprintf(Cout,
				"/* %4d */\t0x%08xU%s",
				i,
				cpool[i].ud[0],
				((i == cpoolsp-1) ? "\n" : ",\n"));
		}

		fprintf(Cout, "};\n");
		fflush(Cout);
	}
}

static void
cpooldump64(void)
{
	if (cpoolsp) {
		register int i;

		fprintf(Cout, "\n/* SWAR constant pool */\n");
		fprintf(Cout, "p64_t cpool64[] = {\n");

		for (i=0; i<cpoolsp; ++i) {
			fprintf(Cout,
				"/* %4d */\t{0x%016llxULL}%s",
				i,
				cpool[i].q[0],
				((i == cpoolsp-1) ? "\n" : ",\n"));
		}

		fprintf(Cout, "};\n");
		fflush(Cout);
	}
}

static void
cpooldump128(void)
{
	if (cpoolsp) {
		register int i;

		fprintf(Cout, "\n/* SWAR constant pool */\n");
		fprintf(Cout, "p128_t cpool128[] = {\n");

		for (i=0; i<cpoolsp; ++i) {
			fprintf(Cout,
				"/* %4d */\t{{0x%016llxULL, 0x%016llxULL}}%s",
				i,
				cpool[i].q[1],
				cpool[i].q[0],
				((i == cpoolsp-1) ? "\n" : ",\n"));
		}

		fprintf(Cout, "};\n");
		fflush(Cout);
	}
}

void
cpooldump(void)
{
	switch (bitsperfrag()) {
	case 128:
		cpooldump128();
		break;
	case 64:
		cpooldump64();
		break;
	case 32:
		cpooldump32();
		break;
	default:
		{
			char buf[50];
			snprintf(buf,
				 50,
				 "cpooldump() doesn't handle %d-bit frags",
				 bitsperfrag());
			bug(buf);
		}
	}
}

