/*	ir.c

	Internal Representation basic builder ops
*/


#include "stdpccts.h"
#include "swartypes.h"
#include "ir.h"
#include "coercer.h"
#include "fe_optimizer.h"
#include "sym.h"
#include "oputils.h"
#include "messages.h"
#include "showir.h"

int	numbuf[MAXNUMS];
int	numsp = 0;

void
string2vnum(register char *p)
{
	while (*p) {
		switch (*p) {
		case '\\':
			switch (*(++p)) {
			case 'n':	numbuf[numsp++] = '\n'; ++p; break;
			case 't':	numbuf[numsp++] = '\t'; ++p; break;
			case 'b':	numbuf[numsp++] = '\b'; ++p; break;
			case 'r':	numbuf[numsp++] = '\r'; ++p; break;
			case 'f':	numbuf[numsp++] = '\f'; ++p; break;
			default:
				if ((*p >= '0') && (*p <= '7')) {
					numbuf[numsp] = *(p++) - '0';
					if ((*p >= '0') && (*p <= '7')) {
						numbuf[numsp] *= 8;
						numbuf[numsp] += *(p++) - '0';
						if ((*p >= '0') && (*p <= '7'))
						{
							numbuf[numsp] *= 8;
							numbuf[numsp] +=
								*(p++) - '0';
						}
					}
				} else {
					numbuf[numsp] = *(p++);
				}
				++numsp;
			}
			break;
		default:
			numbuf[numsp++] = *(p++);
		}
	}
}


int wherestk[256] = {0};	/* what masking is in effect? */
int wheresp = 0;

int breakstk[256] = {0};	/* where does break go? */
int breaksp = 0;

int continuestk[256] = {0};	/* where does continue go? */
int continuesp = 0;

char *
wheremask(register int lev)
{
	/* name of where mask at level lev */
	static char wheremaskbuf[32];

	snprintf(&(wheremaskbuf[0]), 32, "_where%d", lev);
	return(&(wheremaskbuf[0]));
}

static int newlabelnum = 0;

int
newlabels(register int count)
{
	/* reserve count new labels */
	register int n = newlabelnum;
	newlabelnum += count;
	return(n);
}



/*	To make an attribute value, every token's text is copied
	to a dynamically-allocated string cache, so that it may
	freely referenced without worrying about the strings
	being overwritten.  To save space, there is only one
	copy of each string, hence the cache.  No big deal.
*/

static
struct strcache {
	struct strcache *next;
	char text[1];
} strcachehead = { 0, {'\000'} };

void
zzcr_attr(Attrib *a, int tok, char *s)
{
	register struct strcache *p = &strcachehead;
	register struct strcache *q;

	while (p->next) {
		p = p->next;
		if (0 == strcmp(&(p->text[0]), s)) {
			a->text = &(p->text[0]);
			return;
		}
	}

	q = ((struct strcache *) malloc(sizeof(*q) + strlen(s)));
	q->next = 0;
	strcpy(&(q->text[0]), s);
	p->next = q;
	a->text = &(q->text[0]);
}



/*	Stuff for building the tree IR....
*/

typ typnull = { 0, 0, 0 };		/* type is implied by use */
typ typconst = { TYP_INT, 32, 1 };	/* ordinary int constant */
typ typfloat = { TYP_FLOAT, 32, 1 };	/* ordinary float constant */

typ typ1 = { TYP_SWAR, 1, 1 };
typ typ2 = { TYP_SWAR, 2, 1 };
typ typ4 = { TYP_SWAR, 4, 1 };
typ typ8 = { TYP_SWAR, 8, 1 };
typ typ16 = { TYP_SWAR, 16, 1 };
typ typ32 = { TYP_SWAR, 32, 1 };
typ typ64 = { TYP_SWAR, 64, 1 };
typ typ128 = { TYP_SWAR, 128, 1 };

typ typ1ss = { TYP_SWAR|TYP_SAT, 1, 1 };
typ typ2ss = { TYP_SWAR|TYP_SAT, 2, 1 };
typ typ4ss = { TYP_SWAR|TYP_SAT, 4, 1 };
typ typ8ss = { TYP_SWAR|TYP_SAT, 8, 1 };
typ typ16ss = { TYP_SWAR|TYP_SAT, 16, 1 };
typ typ32ss = { TYP_SWAR|TYP_SAT, 32, 1 };
typ typ64ss = { TYP_SWAR|TYP_SAT, 64, 1 };
typ typ128ss = { TYP_SWAR|TYP_SAT, 128, 1 };

typ typ1u = { (TYP_UNSIGN | TYP_SWAR), 1, 1 };
typ typ2u = { (TYP_UNSIGN | TYP_SWAR), 2, 1 };
typ typ4u = { (TYP_UNSIGN | TYP_SWAR), 4, 1 };
typ typ8u = { (TYP_UNSIGN | TYP_SWAR), 8, 1 };
typ typ16u = { (TYP_UNSIGN | TYP_SWAR), 16, 1 };
typ typ32u = { (TYP_UNSIGN | TYP_SWAR), 32, 1 };
typ typ64u = { (TYP_UNSIGN | TYP_SWAR), 64, 1 };
typ typ128u = { (TYP_UNSIGN | TYP_SWAR), 128, 1 };

typ typ1us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 1, 1 };
typ typ2us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 2, 1 };
typ typ4us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 4, 1 };
typ typ8us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 8, 1 };
typ typ16us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 16, 1 };
typ typ32us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 32, 1 };
typ typ64us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 64, 1 };
typ typ128us = { (TYP_UNSIGN | TYP_SWAR | TYP_SAT), 128, 1 };

typ typ32f = { (TYP_FLOAT | TYP_SWAR), 32, 1 };


tree *
mk_node(tree *down,
tree *right,
int op,
sym *symbol,
int num,
typ type)
{
	register tree *t = (tree *) malloc(sizeof(tree));

	t->down = down;
	t->right = right;
	t->op = op;
	t->symbol = symbol;
	t->num = num;
	t->type = type;
	return(t);
}

tree *
mk_leaf(int op,
sym *symbol,
int num,
typ type)
{
	return(mk_node(((tree *) 0), ((tree *) 0), op, symbol, num, type));
}

tree *
mk_unary(int op,
tree *down)
{
	/* Unary op, type is type of child */
	return(mk_node(down, ((tree *) 0), op, ((sym *) 0), 0, down->type));
}

tree *
mk_binary(int op,
tree *left,
tree *right)
{
	/* Binary op, type is cover type of children */
	register typ t = retyp(left->type, right->type);

	incompat(left->type, right->type);
	left = mk_type(t, left);
	right = mk_type(t, right);
	left->right = right;
	return(mk_node(left,
		       ((tree *) 0),
		       op,
		       ((sym *) 0),
		       0,
		       t));
}

tree *
mk_trinary(int op,
tree *one,
tree *two,
tree *twe)
{
	/* Trinary op, type is cover of all args */
	register typ t = retyp(one->type, retyp(two->type, twe->type));

	incompat(one->type, two->type);
	incompat(one->type, twe->type);
	twe = mk_type(t, twe);
	two = mk_type(t, two);
	two->right = twe;
	one->right = two;
	return(mk_node(one,
		       ((tree *) 0),
		       op,
		       ((sym *) 0),
		       0,
		       t));
}

tree *
mk_cast(typ type,
tree *down)
{
	register tree *t;

	t = mk_node(down, down->right, CAST, ((sym *) 0), 0, type);
	down->right = 0;
	return(t);
}

tree *
mk_type(typ type,
tree *down)
{
	if (diftyp(down->type, type)) {
		return(mk_cast(type, down));
	} else {
		return(down);
	}
}

tree *
mk_cloned(tree *orig)
{
	/* Clone tree node "orig" */

	return mk_leaf(orig->op, orig->symbol, orig->num, orig->type);
}

tree *
mk_clone(tree *t)
{
	/* Clone a tree rooted at node "t" */

	register tree *newt;
	register tree *n;

	/* Don't go wild if passed an empty tree */
	if (t == 0) return(0);

	newt = mk_leaf(t->op, t->symbol, t->num, t->type);
	if (t->down) {
		newt->down = mk_clone(t->down);
		n = newt->down;
		t = t->down;
		while (t->right) {
			n->right = mk_clone(t->right);
			n = n->right;
			t = t->right;
		}
	}

	return(newt);
}

tree *
mk_nestwhere(tree *cond)
{
	/* Not nested */
	if ((wheresp < 2) ||
	    (wherestk[wheresp - 1] < 2)) {
		register sym *s;
		register tree *t;
		register int w;

		if (!(s=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		}

		w = wherestk[wheresp];

		wherestk[wheresp] = 0;
		t = mk_binary(NE,
			      cond,
			      mk_leaf(NUM,
				      ((sym *) 0),
				      0,
				      typconst));
		t = co_fold(t);
		t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
			      t);
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}

	/* Warn if wrong parallelism width */
	if (wherestk[wheresp - 1] != wherestk[wheresp]) {
		char buf[256];
		snprintf(&(buf[0]),
			256,
			"where enable nesting of [%d] within [%d]",
			cond->type.dim,
			wherestk[wheresp]);
		error(&(buf[0]));
		return(cond);
	}

	/* New where is restriction on previous where */
	{
		register sym *s0;
		register sym *s1;
		register tree *t;
		register tree *mask;
		register int w;

		if (!(s0=lookup(wheremask(wheresp-1)))) {
			typ t = curtyp;
			curtyp = typnull;
			s0 = enter(wheremask(wheresp-1), SYM_USED);
			curtyp = t;
		} 
		if (!(s1=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s1 = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		mask = mk_leaf(LOAD, s0, 0, s0->type);
		w = wherestk[wheresp];

		/* Are mask fields large enough? */
		if (s0->type.bits < s1->type.bits) {
			/* Nope.  Make them correct size */
			mask = mk_binary(NE,
					 mk_cast(s1->type,
						 mask),
					 mk_leaf(NUM,
						 ((sym *) 0),
						 0,
						 typconst));
		}

		wherestk[wheresp] = 0;
		cond = mk_binary(AND,
				 cond,
				 mask);
		t = mk_binary(NE,
			      cond,
			      mk_leaf(NUM,
				      ((sym *) 0),
				      0,
				      typconst));
		t = mk_assign(mk_leaf(LOAD, s1, 0, s1->type),
			      t);
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}
}

tree *
mk_elsewhere(void)
{
	/* Not nested */
	if ((wheresp < 2) ||
	    (wherestk[wheresp - 1] < 2)) {
		register sym *s;
		register tree *t;
		register int w;

		if (!(s=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		w = wherestk[wheresp];

		wherestk[wheresp] = 0;
		t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
			      mk_unary(NOT, mk_leaf(LOAD, s, 0, s->type)));
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}

	/* New where is restriction on previous where */
	{
		register sym *s0;
		register sym *s1;
		register tree *t;
		register tree *cond;
		register tree *mask;
		register int w;

		if (!(s0=lookup(wheremask(wheresp-1)))) {
			typ t = curtyp;
			curtyp = typnull;
			s0 = enter(wheremask(wheresp-1), SYM_USED);
			curtyp = t;
		} 
		if (!(s1=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s1 = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		cond = mk_unary(NOT, mk_leaf(LOAD, s1, 0, s1->type));
		mask = mk_leaf(LOAD, s0, 0, s0->type);
		w = wherestk[wheresp];

		/* Are mask fields large enough? */
		if (s0->type.bits < s1->type.bits) {
			/* Nope.  Make them correct size */
			mask = mk_binary(NE,
					 mk_cast(s1->type,
						 mask),
					 mk_leaf(NUM,
						 ((sym *) 0),
						 0,
						 typconst));
		}

		wherestk[wheresp] = 0;
		cond = mk_binary(AND,
				 cond,
				 mask);
		t = mk_binary(NE,
			      cond,
			      mk_leaf(NUM,
				      ((sym *) 0),
				      0,
				      typconst));
		t = mk_assign(mk_leaf(LOAD, s1, 0, s1->type),
			      t);
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}
}

tree *
mk_loopwhere(void)
{
	/* Not nested */
	if ((wheresp < 2) ||
	    (wherestk[wheresp - 1] < 2)) {
		register sym *s;
		register tree *t;
		register int w;

		if (!(s=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		w = wherestk[wheresp];

		wherestk[wheresp] = 0;
		t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
			      mk_leaf(NUM, ((sym *) 0), -1, typconst));
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}

	/* Warn if wrong parallelism width */
	if (wherestk[wheresp - 1] != wherestk[wheresp]) {
		char buf[256];
		snprintf(&(buf[0]),
			256,
			"loop width is [%d], but is within [%d]",
			wherestk[wheresp],
			wherestk[wheresp - 1]);
		error(&(buf[0]));

		{
			register sym *s;
			register tree *t;
			register int w;

			if (!(s=lookup(wheremask(wheresp)))) {
				typ t = curtyp;
				curtyp = typnull;
				s = enter(wheremask(wheresp), SYM_USED);
				curtyp = t;
			} 
			w = wherestk[wheresp];

			wherestk[wheresp] = 0;
			t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
				      mk_leaf(NUM, ((sym *) 0), -1, typconst));
			t = co_fold(t);
			wherestk[wheresp] = w;
			return(t);
		}
	}

	/* Copy previous where */
	{
		register sym *s0;
		register sym *s1;
		register tree *t;
		register tree *mask;
		register int w;

		if (!(s0=lookup(wheremask(wheresp-1)))) {
			typ t = curtyp;
			curtyp = typnull;
			s0 = enter(wheremask(wheresp-1), SYM_USED);
			curtyp = t;
		} 
		if (!(s1=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s1 = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		mask = mk_leaf(LOAD, s0, 0, s0->type);
		w = wherestk[wheresp];

		/* Are mask fields large enough? */
		if (s0->type.bits < s1->type.bits) {
			/* Nope.  Make them correct size */
			mask = mk_binary(NE,
					 mk_cast(s1->type,
						 mask),
					 mk_leaf(NUM,
						 ((sym *) 0),
						 0,
						 typconst));
		}

		wherestk[wheresp] = 0;
		t = mk_assign(mk_leaf(LOAD, s1, 0, s1->type),
			      mask);
		t = co_fold(t);
		wherestk[wheresp] = w;
		return(t);
	}
}

static tree *
mk_aswhere(tree *dest,
tree *src)
{
	/* If not within a where, nothing to do */
	if (wherestk[wheresp] < 2) return(src);

	/* If dest not a variable, return */
	if (dest->symbol == ((sym *) 0)) return(src);

	/* If dest is scalar, nothing to do */
	if ((dest->symbol)->type.dim < 2) return(src);

	/* Warn if wrong parallelism width */
	if ((dest->symbol)->type.dim != wherestk[wheresp]) {
		char buf[256];
		snprintf(&(buf[0]),
			256,
			"where enable context is [%d], but %s is [%d]",
			wherestk[wheresp],
			(dest->symbol)->text,
			(dest->symbol)->type.dim);
		warn(&(buf[0]));
		return(src);
	}

	/* Need to mask using _where */
	{
		register sym *s;
		register tree *t;
		register tree *mask;

		if (!(s=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		mask = mk_leaf(LOAD, s, 0, s->type);

		/* Are mask fields large enough? */
		if ((dest->symbol) &&
		    (s->type.bits < (dest->symbol)->type.bits)) {
			/* Nope.  Make them correct size */
			mask = mk_binary(NE,
					 mk_cast((dest->symbol)->type,
						 mask),
					 mk_leaf(NUM,
						 ((sym *) 0),
						 0,
						 typconst));
		}

		t = mk_binary(AND,
			      mk_clone(dest),
			      mk_unary(NOT, mk_clone(mask)));
		t = mk_binary(OR,
			      t,
			      mk_binary(AND,
					src,
					mask));
		t = co_fold(t);
		return(t);
	}
}

tree *
mk_assign(tree *dest,
tree *src)
{
	incompat(dest->type, src->type);

	/* Handle where masking */
	src = mk_aswhere(dest, src);

	if ((dest->type.dim == 1) &&
	    (src->type.dim > 1)) {
		char buf[512];
		snprintf(buf,
			512,
			"cannot store [%d] into a scalar",
			src->type.dim);
		error(buf);
		src = mk_reduce(REDUCEOR, src);
	}

	switch (dest->op) {
	case LOADX:
		dest->op = STOREX;
		if ((dest->down)->type.dim == 1) {
			dest->type = typconst;
		}
		dest->type.bits = dest->symbol->type.bits;
		(dest->down)->right = mk_type(dest->type, src);
		break;
	case LOAD:
		dest->op = STORE;
		dest->down = mk_type(dest->type, src);
		break;
	default:
		dest = src;
		error("only variables can be assigned values");
	}
	return(dest);
}

tree *
mk_asop(int op,
int redop,
tree *dest,
tree *src)
{
	if ((dest->type.dim == 1) &&
	    (src->type.dim > 1)) {
		/* Insert reduction... */
		src = mk_reduce(redop, src);
		info(1, "assignment operands cause implicit reduction");
	}

	/* Now do the usual simple op and assign */
	src = mk_binary(op, mk_clone(dest), src);
	return(mk_assign(dest, src));
}

static tree *
mk_redwhere(int op,
tree *down)
{
	/* If not within a where, nothing to do */
	if (wherestk[wheresp] < 2) return(down);

	/* Warn if wrong parallelism width */
	down = co_fold(down);
	if (down->type.dim != wherestk[wheresp]) {
		char buf[256];
		snprintf(&(buf[0]),
			256,
			"where enable context is [%d], "
			"but reduction is on [%d]",
			wherestk[wheresp],
			down->type.dim);
		warn(&(buf[0]));
		return(down);
	}

	/* Need to mask using _where */
	{
		register sym *s;
		register tree *t;
		register tree *mask;
		register int padval;

		if (!(s=lookup(wheremask(wheresp)))) {
			typ t = curtyp;
			curtyp = typnull;
			s = enter(wheremask(wheresp), SYM_USED);
			curtyp = t;
		} 
		mask = mk_leaf(LOAD, s, 0, s->type);
		padval = 0;

		/* Are mask fields large enough? */
		if (s->type.bits < down->type.bits) {
			/* Nope.  Make them correct size */
			mask = mk_binary(NE,
					 mk_cast(down->type,
						 mask),
					 mk_leaf(NUM,
						 ((sym *) 0),
						 0,
						 typconst));
		}

		/* Build padding vector */
		switch (op) {
		case REDUCEAND:	padval = -1; break;
		case REDUCEMAX:	padval = (1 << (down->type.bits - 2));
				break;
		case REDUCEMIN:	padval = (1 << (down->type.bits - 2)) - 1;
				break;
		case REDUCEMUL:	padval = 1; break;
		default:	padval = 0; break;
		}

		if (padval) {
			t = mk_binary(AND,
				      mk_leaf(NUM,
					      ((sym *) 0),
					      padval,
					      typconst),
				      mk_unary(NOT, mk_clone(mask)));
			t = mk_binary(OR,
				      t,
				      mk_binary(AND,
						down,
						mask));
		} else {
			/* Just zero the disabled fields */
			t = mk_binary(AND, down, mask);
		}

		t = co_fold(t);
		return(t);
	}
}

tree *
mk_reduce(int op,
tree *down)
{
	/* Doesn't make sense to reduce a single value */
	if (down->type.dim <= 1) {
		warn("reduction operator applied to a single value");
		return(down);
	}

	/* Take care of where masking */
	down = mk_redwhere(op, down);

	/* Reduction unary op, result type is typconst */
	return(mk_node(down, ((tree *) 0), op, ((sym *) 0), 0, typconst));
}

tree *
mk_newlabel(register int use,
register int num)
{
	char buf[32];

	curtyp.attr = TYP_LAB;
	curtyp.bits = 0;
	curtyp.dim = 0;
	snprintf(&(buf[0]), 32, "_%d", num);
	return(mk_leaf(use, enter(&(buf[0]), SYM_UNUSED), 0, curtyp));
}

