/*	main.c

	Main program...  not really much here.

	Parses command line options
	drives compilation of source files
	and
	sets up output files
*/

#define	SCC_VERSION	"v000730"

#include <unistd.h>	/* fork(), execv(), unlink() */

#include "stdpccts.h"
#include "swartypes.h"	/* Type defs needed in other headers */
#include "messages.h"	/* Error and warning messages and related vars */
#include "showir.h"	/* Functions and def'ns for IR debugging output */
#include "output.h"	/* Functions for output */
#include "cpool.h"	/* cpooldump() */
#include "spool.h"	/* spooldump() */


FILE	*Cout = 0;
FILE	*Hout = 0;
FILE	*Tout = 0;

char	*sourcename = "";
char	*funcname = "";
char	*Coutput = "Scout.c";
char	*Houtput = "Scout.h";

int	fieldcount = 0;
int	fieldtotal = 0;
int	clocktotal = 0;


/* Default CPU type (initialized from Makefile) */
int	optcpu = DEFAULT_TARGET;


int	optlines = 1;	/* option to generate line directives */
int	opttime = 0;	/* option to take this many seconds */
int	optperf = 0;	/* option to output performance stats */
int	optversion = 0;	/* option to output version info and exit */
int	optverb = 0;	/* option to output verbose info */
int	optir = 0;	/* option to output a rep. of the IR */

int	optnofecf = 0;		/* option to NOT perform co_folds in the
				   front-end */
int	optfebvt = 0;		/* option to perform BVT on vectors in the
				   front-end */
int	optnobecf = 0;		/* option to NOT perform cofolds in the
				   back-end */
int	optnobepeep = 0;	/* option to NOT perform peephole optimizations
				   in the back-end */

static	int cpp_only = 0;	/* only run preprocessor */
static	char *cpp_argv[512];	/* args to cpp */
static	int cpp_argc = 1;	/* how many? */

static	char *gcc_argv[512];	/* args to gcc */
static	int gcc_argc = 1;	/* how many? */
static	int run_gcc = 1;	/* run gcc if we can */

static	char *src_files[512];	/* source files from command line */
static	int src_argc = 0;	/* how many? */

static	int keepc = 0;		/* keep generated .c file (Coutput) */
static	int keeph = 0;		/* keep generated .h file (Houtput) */
static	int keepSi = 0;		/* keep generated .Si file(s) */

#include <sys/types.h>
#include <sys/wait.h>


static char *
targetname(void)
{
	switch (optcpu) {
	case GenericIA32:	return("generic IA32");
	case _6x86MX:		return("Cyrix 6x86MX");
	case K6:		return("AMD K6 MMX");
	case K6_2:		return("AMD K6-2 MMX 3DNow!");
	case K6_3:		return("AMD K6-3 MMX 3DNow!");
	case Athlon:		return("AMD Athlon");
	case AltiVec:		return("PowerPC AltiVec");
	case MII:		return("Cyrix MII");
	case PentiumII:		return("Intel PentiumII MMX");
	case PentiumMMX:	return("Intel Pentium MMX");
	case PARISC2:		return("HP PA-RISC 2.0 class w/MAX");
	}
	return("generic IA32 MMX");
}

static void
openoutputs(void)
{
	if (Cout == 0) {
		Cout = fopen(Coutput, "w");
		if (Cout == 0) {
			fprintf(stderr,
				"error: could not open C output file %s\n",
				Coutput);
			exit(1);
		}

		Hout = fopen(Houtput, "w");
		if (Cout == 0) {
			fprintf(stderr,
				"error: could not open header output file %s\n",
				Houtput);
			exit(1);
		}

		{
			char buf[128];
			snprintf(buf, 128, "Scc compiling for %s target",
				targetname());
			info(1, buf);
		}

		/* Output compilation information */
		fprintf(Cout,
			"/* compiled by Scc alpha-test %s for %s target */\n",
			SCC_VERSION,
			targetname());
		fprintf(Hout,
			"/* compiled by Scc alpha-test %s for %s target */\n",
			SCC_VERSION,
			targetname());

		/* Include the header files for this target */
		headers();
	}
}

static void
swarc2c(register char *src)
{
	/* translate (or at least copy) the file named src */
	static char srcbuf[1024];
	FILE *sourcefile;

	/* does this file need preprocessing? */
	if ((strcmp(".Sc", (src + strlen(src) - 3)) == 0) ||
	    (strcmp(".Sh", (src + strlen(src) - 3)) == 0)) {
		int child;
		int status;

		/* change from .Sc to .Si for preprocessed version */
		strcpy(srcbuf, src);
		strcpy((srcbuf + strlen(src) - 3), ".Si");
		cpp_argv[0] = CPP;
		cpp_argv[cpp_argc + 0] = src;
		cpp_argv[cpp_argc + 1] = srcbuf;
		cpp_argv[cpp_argc + 2] = 0;

		if (optverb) {
			register int i;

			fprintf(stderr, CPP);
			for (i=1; cpp_argv[i]; ++i) {
				fprintf(stderr, " %s", cpp_argv[i]);
			}
			fprintf(stderr, "\n");
		}

		child = fork();
		if (child == 0) {
			execv(CPP, &(cpp_argv[0]));
			fprintf(stderr,
				"internal error: cannot exec C preprocessor "
				"%s\n",
				CPP);
			exit(1);
		}
		waitpid(child, &status, 0);
		if ((WIFEXITED(status) == 0) ||
		    (WEXITSTATUS(status) != 0)) {
			fprintf(stderr,
				"error: preprocessing of %s failed\n",
				sourcename);
			exit(1);
		}

		/* Is this all we were supposed to do? */
		if (cpp_only) return;

		sourcename = srcbuf;
	} else {
		sourcename = src;
	}

	/* if this is SWARC code, compile this file... */
	if ((strcmp(".Sc", (sourcename + strlen(sourcename) - 3)) == 0) ||
	    (strcmp(".Sh", (sourcename + strlen(sourcename) - 3)) == 0) ||
	    (strcmp(".Si", (sourcename + strlen(sourcename) - 3)) == 0)) {
		register char *s = sourcename;

		if (optverb) {
			fprintf(stderr, "Scc compiling %s...\n", sourcename);
		}

		if ((sourcefile = fopen(sourcename, "r")) == 0) {
			fprintf(stderr,
				"error: cannot open source file %s\n",
				sourcename);
			exit(1);
		}

		/* make sure that the output files are created and open */
		openoutputs();

		/* translate this file */
		funcname = "";
		ANTLR(swarc(), sourcefile);
		fclose(sourcefile);

		/* get rid of .Si file if we generated it */
		if ((!keepSi) && (src != s)) {
			unlink(s);
		}
		return;
	}

	/* if this is ordinary C code, copy it into Cout */
	if (strcmp(".c", (sourcename + strlen(sourcename) - 2)) == 0) {
		register int c;

		if (optverb) {
			fprintf(stderr, "copying %s...\n", sourcename);
		}

		openoutputs();
		if (optlines) {
			fprintf(Cout, "/* 1 \"%s\" */\n", sourcename);
		}
		
		/* Copy the C code into the C output file */
		if ((sourcefile = fopen(sourcename, "r")) == 0) {
			fprintf(stderr,
				"error: cannot open source file %s\n",
				sourcename);
			exit(1);
		}
		while ((c = fgetc(sourcefile)) != EOF) {
			fputc(c, Cout);
		}
		fclose(sourcefile);
		return;
	}

	/* if this is a C header code, copy it into Hout */
	if (strcmp(".h", (sourcename + strlen(sourcename) - 2)) == 0) {
		register int c;

		if (optverb) {
			fprintf(stderr, "copying %s...\n", sourcename);
		}

		openoutputs();
		if (optlines) {
			fprintf(Cout, "/* 1 \"%s\" */\n", sourcename);
		}
		
		/* Copy the header code into the header output file */
		if ((sourcefile = fopen(sourcename, "r")) == 0) {
			fprintf(stderr,
				"error: cannot open source file %s\n",
				sourcename);
			exit(1);
		}
		while ((c = fgetc(sourcefile)) != EOF) {
			fputc(c, Hout);
		}
		fclose(sourcefile);
		return;
	}

	/* if this is library, object, preprocessed or raw assembly code,
	   hand it off to gcc... */
	if ((strcmp(".a", (sourcename + strlen(sourcename) - 2)) == 0) ||
	    (strcmp(".o", (sourcename + strlen(sourcename) - 2)) == 0) ||
	    (strcmp(".s", (sourcename + strlen(sourcename) - 2)) == 0) ||
	    (strcmp(".S", (sourcename + strlen(sourcename) - 2)) == 0)) {
		/* Not used until gcc is called... */
		gcc_argv[gcc_argc++] = sourcename;
		return;
	}

	/* if this is anything else, we don't recognize it */
	fprintf(stderr,
		"error: unrecognized file type for %s\n",
		sourcename);
	exit(1);
}

static void
showopts(void)
{
	fprintf(stderr, "Usage: Scc {-options} files\n");
	fprintf(stderr, "-c\t\tcompile but do not link\n");
	fprintf(stderr, "-k\t\tkeep both .c and .h files generated by Scc\n");
	fprintf(stderr, "-kc\t\tkeep .c file generated by Scc\n");
	fprintf(stderr, "-kh\t\tkeep .h file generated by Scc\n");
	fprintf(stderr, "-kSi\t\tkeep .Si file(s) generated by Scc\n");
	fprintf(stderr, "-llibrary\tcompiled library to link with\n");

	fprintf(stderr,
		"-mIA32\t\ttarget is generic IA32%s\n",
		(optcpu == GenericIA32)? " (the default)" : "");
	fprintf(stderr,
		"-mMMX\t\ttarget is generic MMX%s\n",
		(optcpu == GenericMMX)? " (the default)" : "");
	fprintf(stderr,
		"-mK6\t\ttarget is AMD K6 MMX%s\n",
		(optcpu == K6)? " (the default)" : "");
	fprintf(stderr,
		"-mK6-2\t\ttarget is AMD K6-2 MMX with 3DNow!%s\n",
		(optcpu == K6_2)? " (the default)" : "");
	fprintf(stderr,
		"-mK6-3\t\ttarget is AMD K6-3 MMX with 3DNow!%s\n",
		(optcpu == K6_3)? " (the default)" : "");
	fprintf(stderr,
		"-mAthlon\ttarget is AMD Athlon%s\n",
		(optcpu == Athlon)? " (the default)" : "");
	fprintf(stderr,
		"-m6x86MX\ttarget is Cyrix 6x86MX MMX with extended MMX%s\n",
		(optcpu == _6x86MX)? " (the default)" : "");
	fprintf(stderr,
		"-mMII\t\ttarget is Cyrix MII MMX with extended MMX%s\n",
		(optcpu == MII)? " (the default)" : "");
	fprintf(stderr,
		"-mPentiumMMX\ttarget is Intel Pentium with MMX%s\n",
		(optcpu == PentiumMMX)? " (the default)" : "");
	fprintf(stderr,
		"-mPentiumII\ttarget is Intel Pentium II MMX%s\n",
		(optcpu == PentiumII)? " (the default)" : "");
	fprintf(stderr,
		"-mAltiVec\ttarget is PowerPC AltiVec%s\n",
		(optcpu == AltiVec)? " (the default)" : "");
	fprintf(stderr,
		"-mPA-RISC2\ttarget is HP PA-RISC 2.0 class w/MAX%s\n",
		(optcpu == PARISC2)? " (the default)" : "");

	fprintf(stderr, "-o file\t\tname of output executable file\n");
	fprintf(stderr, "-oc file.c\tname of output C source file\n");
	fprintf(stderr, "-oh file.h\tname of output C header file\n");
	fprintf(stderr, "-oo file.o\tname of output object file\n");
	fprintf(stderr, "-os file.s\tname of output assembly file\n");
	fprintf(stderr, "-p\t\tenable performance estimation messages\n");
	fprintf(stderr, "-v\t\toutput verbose version and trace of actions\n");
	fprintf(stderr, "-C\t\tpreprocessor should preserve comments\n");
	fprintf(stderr, "-Dmacro\t\tpass define of macro to preprocessor\n");
	fprintf(stderr, "-E\t\tstop after preprocessing Sc code\n");
	fprintf(stderr, "-Ec\t\tstop after preprocessing C code\n");
	fprintf(stderr, "-I path\t\tpass include path to Sc preprocessor\n");
	fprintf(stderr, "-Ic path\tpass include path to C preprocessor\n");
	fprintf(stderr, "-Lpath\t\tadditional library path to search\n");
	fprintf(stderr, "-Oseconds\toptimizer can run for seconds per block\n");
	fprintf(stderr, "-P\t\tdisable generation of #line directives\n");
	fprintf(stderr, "-S\t\tstop after generating assembly code\n");
	fprintf(stderr, "-Sc\t\tstop after generating C code and header\n");
	fprintf(stderr, "-Umacro\t\tpass undef of macro to preprocessor\n");
	fprintf(stderr, "-W\t\tbump warning level for more messages\n");
	fprintf(stderr, "-ird\t\toutput a rep. of the IR in DN/RT order\n");
	fprintf(stderr, "-irr\t\toutput a rep. of the IR in RT/DN order\n");
	fprintf(stderr, "--no-fe-cofold\tdo not perform constant folds in"
		"the front end\n");
	fprintf(stderr, "--fe-bvt\tperform BVT on vectors in the front end\n");
	fprintf(stderr, "--no-be-cofold\tdo not perform constant folds in"
		"the back end\n");
	fprintf(stderr, "--no-be-peep\tdo not perform peephole optimizations "
		"in the back end\n");
	fprintf(stderr, "file.a\t\tan archive file to include at link time\n");
	fprintf(stderr, "file.c\t\ta C source file, catenated into output C\n");
	fprintf(stderr,
		"file.h\t\ta C header file, catenated into output header\n");
	fprintf(stderr, "file.o\t\tan object file to include at link time\n");
	fprintf(stderr,
	    "file.s\t\tan assembly code file to include at assembly time\n");
	fprintf(stderr, "file.S\t\tan assembly code file to preprocess\n");
	fprintf(stderr, "file.Sc\t\ta Sc source file\n");
	fprintf(stderr, "file.Sh\t\ta Sc header file\n");
	fprintf(stderr, "file.Si\t\ta preprocessed Sc source file\n");
	fprintf(stderr, "--version\toutput Scc version information and exit\n");
}

int
main(argc, argv)
int argc;
char **argv;
{
	register int i;

	for (i=1; i<argc; ++i) {
		if (*(argv[i]) != '-') {
			/* Another source file... */
			src_files[src_argc++] = argv[i];
		} else {
			switch (*(argv[i] + 1)) {
			case 'c':
			case 'l':
				/* Pass these options to gcc */
				gcc_argv[gcc_argc++] = argv[i];
				break;
			case 'k':
				switch (*(argv[i] + 2)) {
				case 'c': keepc = 1; break;
				case 'h': keeph = 1; break;
				case 'S': keepSi = 1; /* Assume -kS* is -kSi */
					  break;
				default:  keepc = 1; keeph = 1; break;
				}
				break;
			case 'm':
				if (strcmp("-mIA32", argv[i]) == 0)
					optcpu = GenericIA32;
				else if (strcmp("-mMMX", argv[i]) == 0)
					optcpu = GenericMMX;
				else if (strcmp("-mK6", argv[i]) == 0)
					optcpu = K6;
				else if (strcmp("-mK6-2", argv[i]) == 0)
					optcpu = K6_2;
				else if (strcmp("-mK6-3", argv[i]) == 0)
					optcpu = K6_3;
				else if (strcmp("-mAthlon", argv[i]) == 0)
					optcpu = Athlon;
				else if (strcmp("-mAltiVec", argv[i]) == 0)
					optcpu = AltiVec;
				else if (strcmp("-mPentiumII", argv[i]) == 0)
					optcpu = PentiumII;
				else if (strcmp("-mPentiumMMX", argv[i]) == 0)
					optcpu = PentiumMMX;
				else if (strcmp("-m6x86MX", argv[i]) == 0)
					optcpu = _6x86MX;
				else if (strcmp("-mMII", argv[i]) == 0)
					optcpu = MII;
				else if (strcmp("-mPA-RISC2", argv[i]) == 0)
					optcpu = PARISC2;
				/* Otherwise, leave as the default */
				break;
			case 'P':
				/* Don't generate line directives */
				optlines = 0;
				/* fall through... */
			case 'C':
				/* Pass these options to cpp */
				cpp_argv[cpp_argc++] = argv[i];
				break;
			case 'D':
			case 'U':
				/* Pass these options to cpp and gcc */
				cpp_argv[cpp_argc++] = argv[i];
				gcc_argv[gcc_argc++] = argv[i];
				break;
			case 'I':
			{
				register char *p =
					malloc(strlen(argv[i + 1]) + 3);
				strcpy(p, "-I");
				strcpy(p + 2, argv[i + 1]);

				switch (*(argv[i] + 2)) {
				case 'c':
					/* -Ic means gcc should -I */
					gcc_argv[gcc_argc++] = p;
					break;
				default:
					/* -I means -I the SWARC code */
					cpp_argv[cpp_argc++] = p;
				}

				++i;
				break;
			}
			case 'L':
				/* Pass this option to gcc */
				gcc_argv[gcc_argc++] = argv[i];
				break;
			case 'E':
				switch (*(argv[i] + 2)) {
				case 'c':
					/* -Ec means gcc should -E */
					gcc_argv[gcc_argc++] = "-E";
					break;
				default:
					/* -E means -E the SWARC code */
					run_gcc = 0;
					cpp_only = 1;
					keepSi = 1;
				}
				break;
			case 'S':
				switch (*(argv[i] + 2)) {
				case 'c':
					/* -Sc means stop after SWARC to C */
					run_gcc = 0;
					keepc = 1;
					keeph = 1;
					break;
				default:
					/* -S means gcc -S */
					gcc_argv[gcc_argc++] = "-S";
				}
				break;
			case 'W':
				/* Set warning level */
				++warnlev;
				break;
			case 'O':
				/* Set optimization level...
				   really time in seconds for scheduler;
				   this also tells gcc -O2
				*/
				opttime = atoi(argv[i] + 1);
				gcc_argv[gcc_argc++] = "-O2";
				break;

			case 'o':
				/* Set output filename */
				switch (*(argv[i] + 2)) {
				case 'c':
					Coutput = argv[++i];
					break;
				case 'h':
					Houtput = argv[++i];
					break;
				case 'o':
					/* Pass -oo file on to gcc as
					   -c -o file */
					gcc_argv[gcc_argc++] = "-c";
					gcc_argv[gcc_argc++] = "-o";
					gcc_argv[gcc_argc++] = argv[++i];
					break;
				case 's':
					/* Pass -os file on to gcc as
					   -S -o file */
					gcc_argv[gcc_argc++] = "-S";
					gcc_argv[gcc_argc++] = "-o";
					gcc_argv[gcc_argc++] = argv[++i];
					break;
				case '\000':
					/* Pass -o file on to gcc */
					gcc_argv[gcc_argc++] = argv[i];
					gcc_argv[gcc_argc++] = argv[++i];
					break;
				default:
					showopts();
					exit(1);
				}
				break;

			case 'p':
				/* Enable performance estimation messages */
				optperf = 1;
				break;

			case 'v':
				/* Enable verbose version trace */
				optverb = 1;
				break;
			case 'i':
				/* Set output filename */
				switch (*(argv[i] + 2)) {
				case 'r':
					/* -ir means output a rep. of the IR */
					switch (*(argv[i] + 3)) {
					case 'd': /* in DOWN/RIGHT order */
						optir = IR_DNRT;
						break;
					case 'r': /* in RIGHT/DOWN order */
						optir = IR_RTDN;
						break;
					default:
						showopts();
						exit(1);
					}
					break;
				default:
					showopts();
					exit(1);
				}
				break;
			case '-':
				if (strcmp("--no-fe-cofold", argv[i]) == 0)
					optnofecf = 1;
				else if (strcmp("--fe-bvt", argv[i]) == 0)
					optfebvt = 1;
				else if (strcmp("--no-be-cofold", argv[i]) == 0)
					optnobecf = 1;
				else if (strcmp("--no-be-peep", argv[i]) == 0)
					optnobepeep = 1;
				else if (strcmp("--version", argv[i]) == 0)
					optversion = 1;
				break;
			default:
				showopts();
				exit(1);
			}
		}
	}

	/* If version-only requested, announce version and target, then exit */
	if (optversion) {
		fprintf(stderr,
			"Scc is alpha-test version %s targeting %s\n",
			SCC_VERSION,
			targetname());
		exit(0);
	}

	/* If verbose output requested, announce version and target */
	if (optverb) {
		fprintf(stderr,
			"Scc is alpha-test version %s targeting %s\n",
			SCC_VERSION,
			targetname());
	}

	/* If no source files, display usage and exit... */
	if (src_argc < 1) {
		showopts();
		exit(0);
	}

	/* ... otherwise, process the files */
	for (i=0; i<src_argc; ++i) {
		/* Compile this file as per options */
		swarc2c(src_files[i]);
	}


	if (Cout != 0) {
		/* Dump the constant pool */
		if (optverb) {
			fprintf(stderr, "Dumping constant pool\n");
		}
		cpooldump();

		/* Dump the spill pool */
		if (optverb) {
			fprintf(stderr, "Declaring spill pool\n");
		}
		spooldump();
	}

	/* Close the SWARC output files */
	if (optverb) {
		fprintf(stderr, "Closing output files\n");
	}
	if (Cout != 0) fclose(Cout);
	if (Hout != 0) fclose(Hout);

	/* Do we want to run gcc now? */
	/* Don't run it if told not to on command line or if too many errors */
	if ((run_gcc == 0) || (errorcount > 0)) {
		/* Nope */
		if (!keepc) unlink(Coutput);
		if (!keeph) unlink(Houtput);
		if (errorcount) {
			fprintf(stderr, "Too many errors.  Exiting...\n");
		}
		exit(errorcount > 0);
	}

	/* Ok, run gcc */
	gcc_argv[0] = GCC;
	gcc_argv[gcc_argc] = Coutput;
	gcc_argv[gcc_argc + 1] = 0;

	/* If verbose output requested, announce execution of gcc */
	if (optverb) {
		register int i;

		fprintf(stderr, GCC);
		for (i=1; gcc_argv[i]; ++i) {
			fprintf(stderr, " %s", gcc_argv[i]);
		}
		fprintf(stderr, "\n");
	}

	/* Exec gcc */
	{
		int child;
		int status;

		child = fork();
		if (child == 0) {
			execv(GCC, &(gcc_argv[0]));
			fprintf(stderr,
				"internal error: cannot exec gcc compiler\n");
			exit(1);
		}
		waitpid(child, &status, 0);
		if ((WIFEXITED(status) == 0) ||
		    (WEXITSTATUS(status) != 0)) {
			fprintf(stderr,
				"error: gcc compile of C output %s failed\n",
				sourcename);
			exit(1);
		}
	}

	/* get rid of the files we generated if we are supposed to */
	if (!keepc) unlink(Coutput);
	if (!keeph) unlink(Houtput);
	exit(0);
}

