/*
	swarc.g

	Parses SWARC code,
	generates function prototypes and definition headings,
	calls back-end (via cg-tree) to generate function body code.
*/

#header <<
	#include "swarc.h"
	#include <string.h>
	#include "ir.h"
>>

<<
	#include "messages.h"
	#include "showir.h"
	#include "sym.h"
	#include "fe_optimizer.h"
	#include "fragmenter.h"
	#include "output.h"
>>

#lexclass START
#token "[\ \t]+"	<< zzskip(); >>
#token "\n"		<< ++zzline; zzskip(); >>
#token "/\*"		<< zzreplstr("");
			   zzmode(LEXCOM);
			   zzmore();
			>>
#token "\""		<< zzreplstr("");
			   zzmode(LEXSTRING);
			   zzmore();
			>>
#token "\'"		<< zzreplstr("");
			   zzmode(LEXCHAR);
			   zzmore();
			>>
#token CSTART "$\{"	<< ++zzline;
			   zzreplstr("");
			   zzmode(LEXC);
			>>
/*
#token "$\{"		<< ++zzline;
			   zzreplstr("");
			   zzmode(LEXC);
			   zzmore();
			>>
*/
#token "#[\ \t]*[0-9]+[\ \t]*"	<<
			   zzline = atoi(zzlextext + 1);
			   zzreplstr("");
			   zzmode(LEXINC);
			   zzmore();
			>>

#lexclass LEXCOM
#token "\n"		<< ++zzline;
			   zzreplstr("");
			   zzmore();
			>>
#token "\*/"		<< zzmode(START);
			   zzskip();
			>>
#token "~[]"		<< zzreplstr("");
			   zzmore();
			>>

#lexclass LEXSTRING
#token "\n"		<< ++zzline; zzmore(); >>
#token "\\\n"		<< ++zzline; zzmore(); >>
#token "\\~[]"		<< zzmore(); >>
#token String "\""	<< zzreplstr(""); zzmode(START); >>
#token "~[\\\"]"	<< zzmore(); >>

			   
#lexclass LEXCHAR
#token "\n"		<< ++zzline; zzmore(); >>
#token "\\\n"		<< ++zzline; zzmore(); >>
#token "\\~[]"		<< zzmore(); >>
#token Char "\'"	<< zzreplstr(""); zzmode(START); >>
#token "~[\\\"]"	<< zzmore(); >>


#lexclass LEXC
#token CCODE "$\}"	<< zzreplstr(""); zzmode(START); >>
#token "\\$"		<< zzreplstr("$"); zzmore(); >>
#token "\\\\"		<< zzreplstr("\\"); zzmore(); >>
#token "\\"		<< zzmore(); >>
#token "$"		<< zzreplstr("#"); zzmore(); >>
#token "~[$\n\\;]*"	<< zzmore(); >>
/* #token "\n"		<< ++zzline; zzmore(); >> */
#token CLINE "\n"	<< ++zzline; >>
#token CSYNC ";"	<< >>


#lexclass LEXINC
#token "\"~[\"]*\""	<< {
			   	static char buf[513];
				zzlextext[ strlen(zzlextext)-1 ] = 0;
				strcpy(&(buf[0]), zzlextext + 1);
				sourcename = &(buf[0]);
			   }
			   zzreplstr("");
			   /* fprintf(Cout, "\n# %d \"%s\"\n",
				   zzline, sourcename); */
			   fprintf(Cout, "\n/* # %d \"%s\" */\n",
				   zzline, sourcename);
			   zzmore();
			>>
#token "~[\"\n]*\n"	<< zzmode(START); zzskip(); >>

#lexclass START


swarc
:	<<
		/* Initialize the table of undefined symbols */
		/* memset((void *)undeclptr, UNUSED, 4096*sizeof(sym)); */
		/* Initialize function name tracking */
		fname_push ( "At top level" );
	>>
	( /* CCODE */
	  CSTART
	  <<
		funcname = "";
		if (optlines) {
			fprintf(Cout,
				"\n/* # %d \"%s\" */\n",
				 zzline,
				 sourcename);
		}
	  >>
	  (  CLINE
	     <<
		fprintf(Cout, "%s", $1.text);
	     >>
	   |
	     CSYNC
	     <<
		fprintf(Cout, "%s", $1.text);
	     >>
	  )*
	  CCODE
	  <<
		fprintf(Cout, "\n");
	  >>
	 |
	  (
	   <<
		curtyp.attr = 0;
		funcname = "";
	   >>
	   { "extern"
	     << curtyp.attr |= TYP_EXTERN; >>
	    |
	     "static"
	     << curtyp.attr |= TYP_STATIC; >>
	   }
	   (data | func)
	  )
	  << curtyp.attr = 0; >>
	)+
	<< showunusedstatics(); >>
;

data:
	<<
		register sym *symbol;
	>>
	type Name
	<<
		/* Error if redefined in this scope or make entry in symtab */
		if ((symbol=thiscope($2.text))) {
			redeclared(symbol);
		} else {
			enter($2.text, SYM_UNUSED);
		}

		if (curscope == 0) {
			/* output globals immediately */
			if (!(curtyp.attr & TYP_STATIC)) {
				if (optlines) {
					fprintf(Hout, "\n/* # %d \"%s\" */\n",
						zzline, sourcename);
				}
				fprintf(Hout, "extern ");
				p_htype(&curtyp);
				fprintf(Hout, "%s", $2.text);
				p_hdim(&curtyp);
				fprintf(Hout, ";\n");
			}
			if (optlines) {
				fprintf(Cout, "\n/* # %d \"%s\" */\n", zzline,
					sourcename);
			}
			p_ctype(&curtyp);
			fprintf(Cout, "%s", $2.text);
			p_cdim(&curtyp);
			fprintf(Cout, ";\n");
		}
	>>
	("," Name
	<<
		/* Error if redefined in this scope or make entry in symtab */
		if ((symbol=thiscope($2.text))) {
			redeclared(symbol);
		} else {
			enter($2.text, SYM_UNUSED);
		}
		if (curscope == 0) {
			/* output globals immediately */
			if (!(curtyp.attr & TYP_STATIC)) {
				if (optlines) {
					fprintf(Hout, "\n/* # %d \"%s\" */\n",
						zzline, sourcename);
				}
				fprintf(Hout, "extern ");
				p_htype(&curtyp);
				fprintf(Hout, "%s", $2.text);
				p_hdim(&curtyp);
				fprintf(Hout, ";\n");
			}
			if (optlines) {
				fprintf(Cout, "\n/* # %d \"%s\" */\n", zzline,
					sourcename);
			}
			p_ctype(&curtyp);
			fprintf(Cout, "%s", $2.text);
			p_cdim(&curtyp);
			fprintf(Cout, ";\n");
		}
	>>
	)* ";"
;

func
:	<<
		register int typed = 0;
		register sym *symbol;
	>>
	{ "void"
	<<
		typed = 1;
	>>
	}
	<<
		if (!typed) warn("return-type defaults to `void'");
	>>
	Name "\("
	<<
		curtyp.attr |= (TYP_VOID | TYP_FUNC);
		curtyp.dim = 0;
		/* Error if redefined in this scope or make entry in symtab */
		if ((symbol=thiscope($2.text))) {
			redeclared(symbol);
		} else {
			enter($2.text, SYM_UNUSED);
		}
		funcname = $2.text;
		fname_push(funcname);
		if (optlines) {
			fprintf(Hout,
				"\n/* # %d \"%s\"*/\n",
				zzline,
				sourcename);
		}
		fprintf(Hout, "extern ");
		p_htype(&curtyp);
		fprintf(Hout, "%s(", $2.text);
		if (optlines) {
			fprintf(Cout,
				"\n/* # %d \"%s\"*/\n",
				zzline,
				sourcename);
		}
		fprintf(Cout, "__inline__\n");
		p_ctype(&curtyp);
		fprintf(Cout, "%s(", $2.text);
		newscope();
	>>
	("void" 
	<<
		fprintf(Hout, "void");
		fprintf(Cout, "void");
	>>
	 | (
	<< curtyp.attr = 0; >>
	  type Name
	<<
		if ((symbol=thiscope($2.text))) {
			redeclared(symbol);
		} else {
			enter($2.text, SYM_UNUSED);
		}
		p_htype(&curtyp);
		fprintf(Hout, "*%s", $2.text);
		p_ctype(&curtyp);
		fprintf(Cout, "*%s", $2.text);
	>>
	  (","
	<< curtyp.attr = 0; >>
	   type Name
	<<
		if ((symbol=thiscope($3.text))) {
			redeclared(symbol);
		} else {
			enter($3.text, SYM_UNUSED);
		}
		fprintf(Hout, ",\n");
		p_htype(&curtyp);
		fprintf(Hout, "*%s", $3.text);
		fprintf(Cout, ",\n");
		p_ctype(&curtyp);
		fprintf(Cout, "*%s", $3.text);
	>>
	  )*)) "\)" ((
	<<
		fprintf(Hout, ");\n");
		fprintf(Cout, ")\n{\n");
		openfunction();
	>>
	  block
	<<
		closefunction();

		co_fold($1.node);

		if (optfebvt) {
			/* Optionally dump ir tree */
			if (optir) {
				if (optverb) {
					printf("Printing IR tree---\n");
					fflush(stdout);
				}
				p_ir($1.node, optir);
				printf("-------------------\n");
				fflush(stdout);
			}
			bvt_vector($1.node);
		}

		/* Optionally dump ir tree */
		if (optir) {
			if (optverb) {
				printf("Printing IR tree---\n");
				fflush(stdout);
			}
			p_ir($1.node, optir);
			printf("-------------------\n");
			fflush(stdout);
		}

		/* The back-end is run on a per-function-body basis. */
		cg_tree($1.node);
		fprintf(Cout, "}\n\n");
	>>
	 ) | ";"
	<<
		fprintf(Hout, ");\n");
		fprintf(Cout, ");\n");
	>>
	)
	<<
		showunused();
		fname_pop();
		endscope();
	>>
;

type
:	<<	/* Initialize curtyp info... */
		curtyp.bits = 0;
		curtyp.dim = 1;
	>>
	("register"
	<< curtyp.attr |= TYP_REG; >>
	 | "const"
	<< curtyp.attr |= TYP_CONST; >>
	 | "modular"
	<< curtyp.attr &= ~TYP_SAT; >>
	 | "saturation"
	<< curtyp.attr |= TYP_SAT; >>
	 | "signed"
	<< curtyp.attr &= ~TYP_UNSIGN; >>
	 | "unsigned"
	<< curtyp.attr |= TYP_UNSIGN; >>
	)*
	<< curtyp.attr &=
		~(TYP_CHAR | TYP_SHORT | TYP_INT | TYP_LONG | TYP_LLONG |
		  TYP_FLOAT | TYP_SWAR);
	>>
	("char"
	<<
		curtyp.attr |= TYP_CHAR;
		curtyp.bits = 8;
	>>
	 | "short"
	<<
		curtyp.attr |= TYP_SHORT;
		curtyp.bits = 16;
	>>
	 | "int"
	<<
		curtyp.attr |= TYP_INT;
		curtyp.bits = 32;
	>>
	 | "long"
	   ( 
		"long" {"int"}
		<<
			curtyp.attr |= TYP_LLONG;
			curtyp.bits = 64;
		>>
		| {"int"}
		<<
			curtyp.attr |= TYP_LONG;
			curtyp.bits = 32;
		>>
	   ) 
	 | "float"
	<<
		/* Floats are always 32 bits for 3DNow! */
		curtyp.attr |= TYP_FLOAT;
		curtyp.bits = 32;
	>>
	)
	{":"
	<<
		/* It's an integer SWAR type -- only bit length matters */
		if (!(curtyp.attr & TYP_FLOAT)) {
			curtyp.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
					 TYP_LONG | TYP_LLONG);
		}
		curtyp.attr |= TYP_SWAR;
	>>
	 {num
	<<
		if (!(curtyp.attr & TYP_FLOAT)) {
			curtyp.bits = $1.num;
		}
	>>
	 }}
	{"\[" (expr
	<<
		if (($1.node)->op != NUM) {
			error("width must have a constant value");
		} else {
			curtyp.dim = ($1.node)->num;
		}
	>>
	  |
	<< curtyp.dim = 0; >>
	  ) "\]"}
|	"typeof" "\(" expr "\)"
	<< curtyp = ($3.node)->type; >>
;

block
:	"\{"
	<< {
		curtyp = typnull;
		newscope();
		$$.node = mk_leaf(BLOCK, enter("", SYM_USED), 0, typnull);
		($$.node)->line = zzline;
	} >>
	(
	<< curtyp.attr = 0; >>
	 data)* {stat
	<< ($$.node)->down = $1.node; >>
	 (stat
	<< {
		/* tack stat on to end of BLOCK children */
		register tree *p = ($$.node)->down;
		while (p->right) p = p->right;
		p->right = $1.node;
	} >>
	)*}
	<<
		showunused();
		endscope();
	>>
	"\}"
;

stat:
	<<
		register sym *symbol;
	>>
	block
	<< $$.node = $1.node; >>
|	/* CCODE */
	CSTART
	<<
	    {
		/* Inline user C code... */
		char *s = "";
		char *t;

		curtyp.attr = TYP_LAB;
		curtyp.bits = 0;
		curtyp.dim = 0;
	>>
	(  CLINE
	   <<
		t = malloc( strlen(s) + strlen($1.text) + 1 );
		strcpy(t, s);
		strcat(t, $1.text);
		s = t;
	   >>
	 |
	   CSYNC
	   <<
		t = malloc( strlen(s) + strlen($1.text) + 1 );
		strcpy(t, s);
		strcat(t, $1.text);
		s = t;
	   >>
	)*
	CCODE
	<<
		$$.node = mk_leaf(CCODE, enter(s, SYM_USED), 0, curtyp);
	    }
	>>
|	Name ":" stat
	<<
		/* This is an LL2 construct */
		if ((symbol=lookup($1.text)) && (symbol->declared)) {
			duplabel(symbol);
		} else if (symbol) {
			symbol->declared = SYM_UNDECLARED;
			symbol->decline = zzline;
		} else {
			curtyp.attr = TYP_LAB;
			curtyp.bits = 0;
			curtyp.dim = 0;
			symbol=enter($1.text, SYM_UNUSED);
		}
		$$.node = mk_leaf(LABEL, symbol, 0, curtyp);
	>>
|	"goto" Name ";"
	<<
		if (!(symbol=lookup($2.text))) {
			curtyp.attr = TYP_LAB;
			curtyp.bits = 0;
			curtyp.dim = 0;
			symbol = enter($2.text, SYM_USED);
			symbol->declared = SYM_UNDECLARED;
		}
		$$.node = mk_leaf(GOTO, symbol, 0, curtyp);
	>>
|	"if" "\(" expr
	<<
		/* Create a new scope with the condition variable */
		newscope();
		curtyp.attr = TYP_SWAR;
		curtyp.bits = 32;
		curtyp.dim = 1;
		$$.node = mk_leaf(BLOCK,
				  enter("_if", SYM_USED),
				  ($3.node)->type.dim,
				  typnull);
		($$.node)->line = zzline;

		/* Set _if condition */
		if (($$.node)->num > 1) {
			/* dim>1 requires store into mask and reduction */
			wherestk[++wheresp] = ($$.node)->num;
			curtyp = ($3.node)->type;
			curtyp.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
					 TYP_LONG | TYP_LLONG);
			curtyp.attr &= ~TYP_EXTERN;
			curtyp.attr |= TYP_SWAR;
			enter(wheremask(wheresp), SYM_USED);
			$3.node = mk_nestwhere($3.node);
			wherestk[wheresp] = 0;
			$3.node = mk_reduce(ANY, $3.node);
			wherestk[wheresp] = ($$.node)->num;
		}

		/* Store into _if */
		$3.node = mk_assign(
			mk_leaf(LOAD, lookup("_if"), 0, typ32),
			$3.node);
		$3.node = co_fold($3.node);
	>>
	"\)" stat
	<<
		$3.node->right = $5.node;
		$$.node->down = mk_node($3.node,
					((tree *) 0),
					IF,
					((sym *) 0),
					0,
					typnull);
	>>
	{"else" stat
	<<
		if ($$.node->num < 2) {
			/* Scalar if with else clause */
			(((($$.node)->down)->down)->right)->right = $2.node;
		} else {
			/* Parallel else is like an if (!_where) */
			register tree *t;

			t = mk_elsewhere();
			wherestk[wheresp] = 0;
			t = mk_reduce(ANY, t);
			wherestk[wheresp] = ($$.node)->num;
			t = mk_assign(mk_leaf(LOAD, lookup("_if"), 0, typ32),
				t);
			t = co_fold(t);
			t->right = $2.node;
			t = mk_node(t, ((tree *) 0), IF, ((sym *) 0), 0,
				typnull);
			(($$.node)->down)->right = t;
		}
	>>
	}
	<<
		/* Close where context for a parallel if */
		if (($$.node)->num > 1) {
			--wheresp;
		}

		/* All is done... except ending the block */
		endscope();
	>>
|	"while" "\(" expr
	<<
		/* Create a new scope with the condition variable */
		newscope();

		/* Considering the way _while is actually used, I think it
		   should be a C type instead of a SWAR type, but masking out
		   the TYP_SWAR here causes the test farther below to fail,
		   which I don't know how to correct off-hand, so for now I'll
		   just change the WHILE case in fragment() to work with the
		   while, for, and do cases here as they are currently written
		   (7-20-2000).
		*/
		curtyp.attr = TYP_SWAR;
		curtyp.bits = 32;
		curtyp.dim = 1;
		$$.node = mk_leaf(BLOCK,
				  enter("_while", SYM_USED),
				  ($3.node)->type.dim,
				  typnull);
		($$.node)->line = zzline;

		/* Allocate break and continue labels */
		$1.num = newlabels(2);
		breakstk[++breaksp] = $1.num + 1;
		continuestk[++continuesp] = $1.num;

		/* Declare and initialize where mask */
		if (($$.node)->num > 1) {
			register tree *t;
			register sym *s;

			wherestk[++wheresp] = ($$.node)->num;
			curtyp = ($3.node)->type;
			curtyp.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
					 TYP_LONG | TYP_LLONG);
			curtyp.attr &= ~TYP_EXTERN;
			curtyp.attr |= TYP_SWAR;
			enter(wheremask(wheresp), SYM_USED);
			t = mk_loopwhere();
			($$.node)->down = mk_unary(EXPR, t);

			/* Loop top label */
			(($$.node)->down)->right = mk_newlabel(LABEL, $1.num);

			/* Now evaluate the condition (mask is applied) */
			t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
				mk_binary(NE,
					mk_cast(s->type, $3.node),
					mk_leaf(NUM,
						((sym *) 0),
						0,
						typconst)));
			t = co_fold(t);
			wherestk[wheresp] = 0;
			t = mk_assign(mk_leaf(LOAD,
					lookup("_while"),
					0,
					typ32),
				mk_reduce(ANY, t));
			t = co_fold(t);
			wherestk[wheresp] = ($$.node)->num;

			/* Insert the expression */
			((($$.node)->down)->right)->right = mk_unary(EXPR, t);
			$2.node = ((($$.node)->down)->right)->right;
		} else {
			/* Ordinary sequential while loop */
			register tree *t;

			/* Loop top label */
			($$.node)->down = mk_newlabel(LABEL, $1.num);

			t = mk_assign(mk_leaf(LOAD,
					lookup("_while"),
					0,
					typ32),
				$3.node);
			t = co_fold(t);
			t = mk_unary(EXPR, t);
			(($$.node)->down)->right = t;
			$2.node = t;
		}

		/* Now output the loop exiting goto */
		{
			register tree *t;

			t = mk_newlabel(WHILE, $1.num + 1);
			($2.node)->right = t;
			$2.node = t;
		}
	>>
	"\)" stat
	<<
		/* Ok, now tack-on stuff to $2.node... */
		($2.node)->right = $5.node;
		($5.node)->right = mk_newlabel(GOTO, $1.num);
		(($5.node)->right)->right = mk_newlabel(LABEL, $1.num + 1);

		/* Cleanup... */
		if (($$.node)->num > 1) {
			--wheresp;
		}
		endscope();
	>>
|	"do"
	<<
		/* Create a new scope with the condition variable */
		newscope();
		curtyp.attr = TYP_SWAR;
		curtyp.bits = 32;
		curtyp.dim = 1;
		$$.node = mk_leaf(BLOCK,
				  enter("_do", SYM_USED),
				  0,
				  typnull);
		($$.node)->line = zzline;

		/* Allocate top, break, and continue labels */
		$1.num = newlabels(3);
		breakstk[++breaksp] = $1.num + 2;
		continuestk[++continuesp] = $1.num + 1;
	>>
	stat "while" "\(" expr "\)" ";"
	<<
		/* Loop top label */
		($$.node)->down = mk_newlabel(LABEL, $1.num);

		/* The statement */
		(($$.node)->down)->right = $2.node;
		$3.node = (($$.node)->down)->right;

		/* The continue label */
		($3.node)->right = mk_newlabel(LABEL, $1.num + 1);
		$3.node = ($3.node)->right;

		/* The expression test */
		if (($5.node)->type.dim > 1) {
			/* For now, plural expressions are just reduced */
			info(1, "do loops do not effect enable masking");
			$5.node = mk_reduce(ANY, $5.node);
		}
		$5.node = mk_assign(mk_leaf(LOAD, lookup("_do"), 0, typ32),
			$5.node);
		$5.node = co_fold($5.node);
		($3.node)->right = mk_unary(EXPR, $5.node);
		$3.node = ($3.node)->right;
		
		/* Goto the loop top */
		($3.node)->right = mk_newlabel(DO, $1.num);
		$3.node = ($3.node)->right;

		/* The break label */
		($3.node)->right = mk_newlabel(LABEL, $1.num + 2);

		/* Cleanup... */
		endscope();
	>>
|	"for" "\(" expr ";" expr ";"
	<<
		/* Create a new scope with the condition variable */
		newscope();
		/* Considering the way _while is actually used, I think it
		   should be a C type instead of a SWAR type, but masking out
		   the TYP_SWAR here causes the test farther below to fail,
		   which I don't know how to correct off-hand, so for now I'll
		   just change the WHILE case in fragment() to work with the
		   while, for, and do cases here as they are currently written
		   (7-20-2000).
		*/
		curtyp.attr = TYP_SWAR;
		curtyp.bits = 32;
		curtyp.dim = 1;
		$$.node = mk_leaf(BLOCK,
				  enter("_while", SYM_USED),
				  ($5.node)->type.dim,
				  typnull);
		($$.node)->line = zzline;

		/* Allocate loop top, break, and continue labels */
		$1.num = newlabels(3);
		breakstk[++breaksp] = $1.num + 2;
		continuestk[++continuesp] = $1.num + 1;

		/* Declare and initialize where mask */
		if (($$.node)->num > 1) {
			register tree *t;
			register sym *s;

			wherestk[++wheresp] = ($$.node)->num;
			curtyp = ($5.node)->type;
			curtyp.attr &=
				~(TYP_CHAR | TYP_SHORT | TYP_INT | TYP_LONG |
				  TYP_LLONG | TYP_FLOAT);
			curtyp.attr &= ~TYP_EXTERN;
			curtyp.attr |= TYP_SWAR;
			s = enter(wheremask(wheresp), SYM_USED);
			t = mk_loopwhere();
			($$.node)->down = mk_unary(EXPR, t);

			/* Loop top label */
			(($$.node)->down)->right = mk_newlabel(LABEL, $1.num);

			/* Now evaluate the condition (mask is applied) */
			t = mk_assign(mk_leaf(LOAD, s, 0, s->type),
				mk_binary(NE,
					mk_cast(s->type, $5.node),
					mk_leaf(NUM,
						((sym *) 0),
						0,
						typconst)));
			t = co_fold(t);
			wherestk[wheresp] = 0;
			t = mk_assign(mk_leaf(LOAD,
					lookup("_while"),
					0,
					typ32),
				mk_reduce(ANY, t));
			t = co_fold(t);
			wherestk[wheresp] = ($$.node)->num;

			/* Insert the expression */
			((($$.node)->down)->right)->right = mk_unary(EXPR, t);
			$2.node = ((($$.node)->down)->right)->right;
		} else {
			/* Ordinary sequential while loop */
			register tree *t;

			/* Loop top label */
			($$.node)->down = mk_newlabel(LABEL, $1.num);

			t = mk_assign(mk_leaf(LOAD,
					lookup("_while"),
					0,
					typ32),
				$5.node);
			t = co_fold(t);
			t = mk_unary(EXPR, t);
			(($$.node)->down)->right = t;
			$2.node = t;
		}

		/* Now output the loop exiting goto */
		{
			register tree *t;

			t = mk_newlabel(WHILE, $1.num + 2);
			($2.node)->right = t;
			$2.node = t;
		}
	>>
	expr "\)" stat
	<<
		/* Now we have some fancier footwork to do to make
		   this really work like for rather than while...
		*/

		/* Paste-in the first expression up front */
		$3.node = mk_unary(EXPR, $3.node);
		($3.node)->right = ($$.node)->down;
		($$.node)->down = $3.node;

		/* At end place statement */
		($2.node)->right = $9.node;
		$2.node = ($2.node)->right;

		/* At end place continuelab */
		($2.node)->right = mk_newlabel(LABEL, $1.num + 1);
		$2.node = ($2.node)->right;

		/* At end place expr3 */
		$7.node = mk_unary(EXPR, $7.node);
		($2.node)->right = $7.node;
		$2.node = ($2.node)->right;

		/* At end place goto top of loop */
		($2.node)->right = mk_newlabel(GOTO, $1.num);
		$2.node = ($2.node)->right;

		/* At end place breaklab */
		($2.node)->right = mk_newlabel(LABEL, $1.num + 2);

		/* Cleanup... */
		if (($$.node)->num > 1) {
			--wheresp;
		}
		endscope();
	>>
|	"continue"
	<<
		$$.num = continuesp;
	>>
	{ expr
	<<
		if (($2.node)->op != NUM) {
			error("continue depth must have a constant value");
		} else if (($2.node)->num > continuesp) {
			error("continue argument exceeds nesting depth");
		} else {
			$$.num = (continuesp + 1 - ($2.node)->num);
		}
	>>
	} ";"
	<<
		$$.node = mk_newlabel(GOTO, continuestk[$$.num]);
	>>
|	"break"
	<<
		$$.num = breaksp;
	>>
	{ expr
	<<
		if (($2.node)->op != NUM) {
			error("break depth must have a constant value");
		} else if (($2.node)->num > breaksp) {
			error("break argument exceeds nesting depth");
		} else {
			$$.num = (breaksp + 1 - ($2.node)->num);
		}
	>>
	} ";"
	<<
		$$.node = mk_newlabel(GOTO, breakstk[$$.num]);
	>>
|	"return" ";"
	<< $$.node = mk_leaf(RETURN, ((sym *) 0), 0, typnull); >>
|	";"
	<< $$.node = mk_leaf(SEMI, ((sym *) 0), 0, typnull); >>
|	"where" "\(" expr
	<<
		if (($3.node)->type.dim < 2) {
			error("where expression must be parallel");
		}

		/* Create a new scope with the condition variable */
		newscope();
		curtyp = ($3.node)->type;
		curtyp.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
				 TYP_LONG | TYP_LLONG | TYP_FLOAT);
		curtyp.attr &= ~TYP_EXTERN;
		curtyp.attr |= TYP_SWAR;
		wherestk[++wheresp] = curtyp.dim;
		$$.node = mk_leaf(BLOCK,
				  enter(wheremask(wheresp), SYM_USED),
				  curtyp.dim,
				  typnull);
		($$.node)->line = zzline;

		/* Make mask just like a parallel if */
		$3.node = mk_nestwhere($3.node);
		$3.node = co_fold($3.node);
		$$.node->down = mk_unary(EXPR, $3.node);
	>>
	"\)" stat
	<<
		(($$.node)->down)->right = $5.node;
	>>
	{"elsewhere" stat
	<<
		/* Need to flip sense of _where mask */
		{
			register tree *t;

			t = mk_elsewhere();
			t = co_fold(t);
			t = mk_unary(EXPR, t);
			t->right = $2.node;
			((($$.node)->down)->right)->right = t;
		}
	>>
	}
	<<
		/* Close where context */
		--wheresp;

		/* All is done... except ending the block */
		endscope();
	>>
|	"everywhere"
	<<
		wherestk[++wheresp] = 0;
	>>
	stat
	<<
		--wheresp;
		$$.node = $2.node;
	>>
|	ident
	<<
		/* Create a new scope for the call arguments */
		newscope();
		curtyp = typnull;
		$$.node = mk_leaf(BLOCK,
				  enter("", SYM_USED),
				  0,
				  typnull);
		($$.node)->line = zzline;
		($$.node)->down = mk_leaf(CALL,
					  $1.symbol,
					  0,
					  typnull);

		if (wherestk[wheresp] > 1) {
			info(1,
			     "called function does not inherit enable masking");
		}
	>>
	"\(" {expr
	<<
		if (($1.node)->op == LOAD) {
			/* pass this argument directly */
			(($$.node)->down)->right = $1.node;
		} else {
			/* store into dummy and pass that */
			register sym *s;
			curtyp = ($1.node)->type;
			s = enter(argname(0), SYM_USED);
			(($$.node)->down)->right =
				mk_unary(EXPR, mk_assign(
				mk_leaf(LOAD, s, 0, s->type),
				$1.node));
		}

		/* bump argument count */
		++((($$.node)->down)->num);
	>>
	 ("," expr
	<< {
		/* find end of arg list */
		register tree *p = (($$.node)->down)->right;
		while (p->right) p = p->right;

		/* tack on another arg */
		if (($2.node)->op == LOAD) {
			/* pass this argument directly */
			p->right = $2.node;
		} else {
			/* store into dummy and pass that */
			register sym *s;
			curtyp = ($2.node)->type;
			s = enter(argname((($$.node)->down)->num), SYM_USED);
			p->right =
				mk_unary(EXPR, mk_assign(
				mk_leaf(LOAD, s, 0, s->type),
				$2.node));
		}

		/* bump argument count */
		++((($$.node)->down)->num);
	} >>
	 )*} "\)" ";"
	<<
		/* now re-link argument list in the right place */
		(($$.node)->down)->down = (($$.node)->down)->right;
		(($$.node)->down)->right = ((tree *) 0);

		/* and cleanup this scope */
		endscope();
	>>
|	expr ";"
	<< $$.node = mk_unary(EXPR, $1.node); >>
;

expr
:	expr0
	<< $$.node = co_fold($1.node); >>
;

expr0
:	expr1
	<< $$.node = $1.node; >>
	{
		"\=" expr
	<<
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"&&=" expr
	<< $$.node = mk_asop(LAND, ALL, $$.node, $2.node); >>
	|	"\|\|=" expr
	<< $$.node = mk_asop(LOR, ANY, $$.node, $2.node); >>
	|	"?\>=" expr
	<< $$.node = mk_asop(MAX, REDUCEMAX, $$.node, $2.node); >>
	|	"?\<=" expr
	<< $$.node = mk_asop(MIN, REDUCEMIN, $$.node, $2.node); >>
	|	"\+/=" expr
	<< $$.node = mk_asop(AVG, REDUCEAVG, $$.node, $2.node); >>
	|	"\+=" expr
	<< $$.node = mk_asop(ADD, REDUCEADD, $$.node, $2.node); >>
	|	"\-=" expr
	<<
		$2.node = mk_binary(SUB, mk_clone($$.node), $2.node);
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"\*=" expr
	<< $$.node = mk_asop(MUL, REDUCEMUL, $$.node, $2.node); >>
	|	"/=" expr
	<<
		$2.node = mk_binary(DIV, mk_clone($$.node), $2.node);
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"%=" expr
	<<
		$2.node = mk_binary(MOD, mk_clone($$.node), $2.node);
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"\>\>=" expr
	<<
		$2.node = mk_binary(SHR, mk_clone($$.node), $2.node);
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"\<\<=" expr
	<<
		$2.node = mk_binary(SHL, mk_clone($$.node), $2.node);
		$$.node = mk_assign($$.node, $2.node);
	>>
	|	"&=" expr
	<< $$.node = mk_asop(AND, REDUCEAND, $$.node, $2.node); >>
	|	"^=" expr
	<< $$.node = mk_asop(XOR, REDUCEXOR, $$.node, $2.node); >>
	|	"\|=" expr
	<< $$.node = mk_asop(OR, REDUCEOR, $$.node, $2.node); >>
	}
;

expr1
:	expr2
	<< $$.node = $1.node; >>
	{ "?" expr2 ":" expr2
	<< $$.node = mk_trinary(QUEST, $$.node, $2.node, $4.node); >>
	}
;

expr2
:	expr3
	<< $$.node = $1.node; >>
	("\|\|" expr3
	<< $$.node = mk_binary(LOR, $$.node, $2.node); >>
	)*
;

expr3
:	expr4
	<< $$.node = $1.node; >>
	("&&" expr4
	<< $$.node = mk_binary(LAND, $$.node, $2.node); >>
	)*
;

expr4
:	expr5
	<< $$.node = $1.node; >>
	("\|" expr5
	<< $$.node = mk_binary(OR, $$.node, $2.node); >>
	)*
;

expr5
:	expr6
	<< $$.node = $1.node; >>
	("^" expr6
	<< $$.node = mk_binary(XOR, $$.node, $2.node); >>
	)*
;

expr6
:	expr7
	<< $$.node = $1.node; >>
	("&" expr7
	<< $$.node = mk_binary(AND, $$.node, $2.node); >>
	)*
;

expr7
:	expr8
	<< $$.node = $1.node; >>
	(
		"\=\=" expr8
	<< $$.node = mk_binary(EQ, $$.node, $2.node); >>
	|	"!=" expr8
	<< $$.node = mk_binary(NE, $$.node, $2.node); >>
	)*
;

expr8
:	expr9
	<< $$.node = $1.node; >>
	(
		"\<" expr9
	<< $$.node = mk_binary(LT, $$.node, $2.node); >>
	|	"\>" expr9
	<< $$.node = mk_binary(GT, $$.node, $2.node); >>
	|	"\<=" expr9
	<< $$.node = mk_binary(LE, $$.node, $2.node); >>
	|	"\>=" expr9
	<< $$.node = mk_binary(GE, $$.node, $2.node); >>
	|	"?\>" expr9
	<< $$.node = mk_binary(MAX, $$.node, $2.node); >>
	|	"?\<" expr9
	<< $$.node = mk_binary(MIN, $$.node, $2.node); >>
	|	"\+/" expr9
	<< $$.node = mk_binary(AVG, $$.node, $2.node); >>
	)*
;

expr9
:	expr10
	<< $$.node = $1.node; >>
	(
		"\>\>" expr10
	<< $$.node = mk_binary(SHR, $$.node, $2.node); >>
	|	"\<\<" expr10
	<< $$.node = mk_binary(SHL, $$.node, $2.node); >>
	)*
;

expr10
:	expr11
	<< $$.node = $1.node; >>
	(
		"\+" expr11
	<< $$.node = mk_binary(ADD, $$.node, $2.node); >>
	|	"\-" expr11
	<< $$.node = mk_binary(SUB, $$.node, $2.node); >>
	)*
;

expr11
:	expr12
	<< $$.node = $1.node; >>
	(
		"\*" expr12
	<< $$.node = mk_binary(MUL, $$.node, $2.node); >>
	|	"/" expr12
	<< $$.node = mk_binary(DIV, $$.node, $2.node); >>
	|	"%" expr12
	<< $$.node = mk_binary(MOD, $$.node, $2.node); >>
	)*
;

expr12
:	<< typ t; >>
	"\-" expr12
	<< $$.node = mk_unary(NEG, $2.node); >>
|	"!" expr12
	<< $$.node = mk_unary(LNOT, $2.node); >>
|	"\~" expr12
	<< $$.node = mk_unary(NOT, $2.node); >>
|	"\+\+" expr12
	<<
		$2.node = mk_binary(ADD,
			mk_clone($$.node = $2.node),
			mk_type(($2.node)->type,
				mk_leaf(NUM, ((sym *) 0), 1, typconst)));
		$$.node = mk_assign($$.node, $2.node);
	>>
|	"\-\-" expr12
	<<
		$2.node = mk_binary(SUB,
			mk_clone($$.node = $2.node),
			mk_type(($2.node)->type,
				mk_leaf(NUM, ((sym *) 0), 1, typconst)));
		$$.node = mk_assign($$.node, $2.node);
	>>
|	"\("
	<< curtyp.attr = 0; >>
	type
	<< t = curtyp; >>
	"\)" expr12
	<<
		/* Cast can have implied dimension, [0] or [] */
		if (t.dim == 0) t.dim = ($4.node)->type.dim;
		$$.node = mk_cast(t, $4.node);
	>>
|	"sizeof" "\(" (type
	<< $$.node = mk_leaf(SIZEOF, enter("", SYM_USED), 0, typconst); >>
	 | ident
	<< $$.node = mk_leaf(SIZEOF, $1.symbol, 0, typconst); >>
	) "\)"
|	expr13
	<< $$.node = $1.node; >>
	{ ("\[" expr "\]"
	<<
		if (($$.node)->op == LOAD) {
			/* (a)[b] is LOADX like a[b] */
			$$.node->op = LOADX;
			$$.node->type.dim = 1;
			($$.node)->down = $2.node;
		} else {
			/* otherwise, (expr)[b] is a PUTGET */
			typ t = typconst;
			$$.node = mk_unary(PUTGET, $$.node);
			t.dim = ($2.node)->type.dim;
			$2.node = mk_type(t, $2.node);
			(($$.node)->down)->right = $2.node;
			if (($2.node)->type.dim == 1) {
				($$.node)->type = typconst;
			}
		}
	>>
	 ) | ( "\[\<\<" expr "\]"
	<<
		if (($2.node)->op != NUM) {
			error("shift offset must be a constant");
		} else {
			if ((($2.node)->num >= ($$.node)->type.dim) ||
			    (($2.node)->num <= -(($$.node)->type.dim))) {
				/* nothing left */
				warn("shift offset beyond array width");
				$$.node = mk_cast(($$.node)->type,
						  mk_leaf(NUM,
						  	  ((sym *) 0),
							  0,
							  typconst));
				$$.node = co_fold($$.node);
			} else if (($2.node)->num) {
				$$.node = mk_node($$.node,
						  ((tree *) 0),
						  SHIFT,
						  ((sym *) 0),
						  ($2.node)->num,
						  ($$.node)->type);
			}
		}
	>>
	 ) | ( "\[\<\<%" expr "\]"
	<<
		if (($2.node)->op != NUM) {
			error("rotate offset must be a constant");
		} else {
			register int off;

			off = ($2.node)->num;
			off %= ($$.node)->type.dim;
			if (off) {
				$$.node = mk_node($$.node,
						  ((tree *) 0),
						  ROTATE,
						  ((sym *) 0),
						  off,
						  ($$.node)->type);
			}
		}
	>>
	 ) | ( "\[\>\>" expr "\]"
	<<
		if (($2.node)->op != NUM) {
			error("shift offset must be a constant");
		} else {
			if ((($2.node)->num >= ($$.node)->type.dim) ||
			    (($2.node)->num <= -(($$.node)->type.dim))) {
				warn("shift offset beyond array width");
				$$.node = mk_cast(($$.node)->type,
						  mk_leaf(NUM,
						  	  ((sym *) 0),
							  0,
							  typconst));
				$$.node = co_fold($$.node);
			} else if (($2.node)->num) {
				$$.node = mk_node($$.node,
						  ((tree *) 0),
						  SHIFT,
						  ((sym *) 0),
						  -(($2.node)->num),
						  ($$.node)->type);
			}
		}
	>>
	 ) | ( "\[\>\>%" expr "\]"
	<<
		if (($2.node)->op != NUM) {
			error("rotate offset must be a constant");
		} else {
			register int off;

			off = ($$.node)->type.dim - ($2.node)->num;
			off %= ($$.node)->type.dim;
			if (off) {
				$$.node = mk_node($$.node,
						  ((tree *) 0),
						  ROTATE,
						  ((sym *) 0),
						  off,
						  ($$.node)->type);
			}
		}
	>>
	) | "\+\+"
	<<
		bug("treating postfix ++ as prefix ++");
		$1.node = mk_binary(ADD,
			mk_clone($$.node),
			mk_type(($$.node)->type,
				mk_leaf(NUM, ((sym *) 0), 1, typconst)));
		$$.node = mk_assign($$.node, $1.node);
	>>
	| "\-\-"
	<<
		bug("treating postfix -- as prefix --");
		$1.node = mk_binary(SUB,
			mk_clone($$.node),
			mk_type(($$.node)->type,
				mk_leaf(NUM, ((sym *) 0), 1, typconst)));
		$$.node = mk_assign($$.node, $1.node);
	>>
	}
|	"&&=" expr12
	<< $$.node = mk_reduce(ALL, $2.node); >>
|	"\|\|=" expr12
	<< $$.node = mk_reduce(ANY, $2.node); >>
|	"?\>=" expr12
	<< $$.node = mk_reduce(REDUCEMAX, $2.node); >>
|	"?\<=" expr12
	<< $$.node = mk_reduce(REDUCEMIN, $2.node); >>
|	"\+/=" expr12
	<< $$.node = mk_reduce(REDUCEAVG, $2.node); >>
|	"\+=" expr12
	<< $$.node = mk_reduce(REDUCEADD, $2.node); >>
|	"&=" expr12
	<< $$.node = mk_reduce(REDUCEAND, $2.node); >>
|	"\*=" expr12
	<< $$.node = mk_reduce(REDUCEMUL, $2.node); >>
|	"\|=" expr12
	<< $$.node = mk_reduce(REDUCEOR, $2.node); >>
|	"^=" expr12
	<< $$.node = mk_reduce(REDUCEXOR, $2.node); >>
;

expr13
:	num
	<< $$.node = mk_leaf(NUM, ((sym *) 0), $1.num, typconst); >>
|	Float_Number
	<< {
		/* Encode float in 32-bit int field */
		union { float f; int i; } t;
		t.f = atof($1.text);
		$$.node = mk_leaf(NUM, ((sym *) 0), t.i, typfloat);
	} >>
|	vnum
	<< $$.node = $1.node; >>
|	ident
	<< {
		typ t = ($1.symbol)->type;

		if (!(optcpu & CPU_AltiVec)) {
			/* Load immediately converts to SWAR type */
			t.attr &= ~(TYP_CHAR | TYP_SHORT | TYP_INT |
				    TYP_LONG | TYP_LLONG);
			t.attr |= TYP_SWAR;
		}
		$$.node = mk_leaf(LOAD, $1.symbol, 0, t);
	} >>
|	"\(" expr "\)"
	<< $$.node = $2.node; >>
;

ident
:	Name
	<<
		if (!($$.symbol=lookup($1.text))) {
			typ t = curtyp;
			curtyp = typnull;
			$$.symbol = undeclared($1.text);
			curtyp = t;
		}
		$$.symbol->used = SYM_USED;
	>>
;

vnum
:	<< $$.node = mk_leaf(VNUM, ((sym *) 0), numsp, typconst); >>
	"\{" expr
	<<
		if (($2.node)->op != NUM) {
			error("constant vector element must have a constant value");
		}
		numbuf[numsp++] = ($2.node)->num;
	>>
	{"\.\." expr
	<<
		if (($2.node)->op != NUM) {
			error("constant vector element must have a constant value");
		}
		while (numbuf[numsp-1] < ($2.node)->num) {
			numbuf[numsp] = (numbuf[numsp-1] + 1);
			++numsp;
			++(($$.node)->type.dim);
		}
	>>
	} ("," expr
	<<
		if (($2.node)->op != NUM) {
			error("constant vector element must have a constant value");
		}
		numbuf[numsp++] = ($2.node)->num;
		++(($$.node)->type.dim);
	>>
	{"\.\." expr
	<<
		if (($2.node)->op != NUM) {
			error("constant vector element must have a constant value");
		}
		while (numbuf[numsp-1] < ($2.node)->num) {
			numbuf[numsp] = (numbuf[numsp-1] + 1);
			++numsp;
			++(($$.node)->type.dim);
		}
	>>
	} )* "\}"
|	String
	<<
		$$.node = mk_leaf(VNUM, ((sym *) 0), numsp, typconst);
		string2vnum($1.text);
	>>
	(String
	<< string2vnum($1.text); >>
	)*
	<<
		/* Only add null terminator after merged string sequence */
		numbuf[numsp++] = 0;
		($$.node)->type.dim = numsp - ($$.node)->num;
	>>
;

num
:	Octal_Number
	<< sscanf($1.text, "%o", &($$.num)); >>
|	Decimal_Number
	<< $$.num = atoi($1.text); >>
|	Hex_Number
	<< sscanf(($1.text + 2), "%x", &($$.num)); >>
|	Binary_Number
	<< {
		/* Also allow binary constants... */
		register char *p = ($1.text + 2);
		$$.num = 0;
		do {
			$$.num = ($$.num << 1) + (*p - '0');
		} while (*++p);
	} >>
|	Char
	<< {
		register int t = numsp;
		register int i = numsp;

		string2vnum($1.text);
		if (numsp > (t + 4)) {
			error("character constant does not fit within a 32-bit word");
			numsp = t + 4;
		}
		$$.num = 0;
		while (i < numsp) {
			$$.num |= (numbuf[i] << (8 * (i - t)));
			++i;
		}
		numsp = t;
	} >>
|	"widthof" "\(" expr "\)"
	<< $$.num = ($3.node)->type.dim; >>
|	"precisionof" "\(" expr "\)"
	<< $$.num = ($3.node)->type.bits; >>
;


#token	Octal_Number	"[0][0-9]*"
#token	Decimal_Number	"[1-9][0-9]*"
#token	Hex_Number	"0[Xx][0-9A-Fa-f]+"
#token	Binary_Number	"0[Bb][01]+"
#token	Float_Number	"([0-9]+\.[0-9]+)|([0-9]+{\.[0-9]*}[Ee]{[\+\-]}[0-9]+)"
#token	Name		"[_A-Za-z][0-9_A-Za-z]*"


/* Trinary operations: */
#token	QUEST
#token	TPERM

/* Binary operations: */
#token	ADD
#token	ADDH
#token	AVG
#token	DIV
#token	EQ
#token	EQ_C
#token	GE
#token	GT
#token	GT_C
#token	AND
#token	ANDN
#token	LAND
#token	LE
#token	LOR
#token	LT
#token	MAX
#token	MIN
#token	MOD
#token	MUL
#token	MULH
#token	MULEVEN
#token	MULODD
#token	NE
#token	NOR
#token	OR
#token	PUTGET
#token	ROTATE
#token	SHIFT
#token	SHL
#token	SHLBIT
#token	SHLBYTE
#token	SHR
#token	SHRBIT
#token	SHRBYTE
#token	SUB
#token	XOR

#token	PACK
#token	PACKS2U
#token	INTRLVLOW
#token	INTRLVHIGH
#token	INTRLVEVEN
#token	INTRLVODD
#token	PERM
#token	REPL

#token	RCP1
#token	RCP2

#token	STORER
#token	STORERR
#token	STOREX


/* Unary operations: */
#token	CAST
#token	LNOT
#token	NEG
#token	NOT

#token	UNPACKL
#token	UNPACKH
#token	RCP
#token	LEA
#token	LOADR
#token	LOADRR
#token	LOADX
#token	STORE
#token	I2F
#token	F2I

/* Nullary operations: */
#token	NUM
#token	VNUM
#token	LOAD
#token	LVSL
#token	SIZEOF

/* Reductions: */
#token	ALL
#token	ANY
#token	REDUCEADD
#token	REDUCEAND
#token	REDUCEAVG
#token	REDUCEMAX
#token	REDUCEMIN
#token	REDUCEMUL
#token	REDUCEOR
#token	REDUCEXOR

/* Constructs: */
#token	BLOCK
#token	BREAK
#token	CALL
#token	CONTINUE
#token	DO
#token	EVERYWHERE
#token	EXPR
#token	FOR
#token	GOTO
#token	IF
#token	LABEL
#token	RETURN
#token	SEMI
#token	WHERE
#token	WHILE
