/*	sym.c

	Stuff for managing the symbol table.  This is a bit
	funky in that it marks each entry not only with the
	scope (depth) number, but also with a lexical block
	serial number.  This old trick allows us to keep the
	whole symbol table around, including items that are no
	longer visible.  For now, this uses slow linear search,
	but it could easily use hashing if speed is a problem.
*/


#include "stdpccts.h"
#include "sym.h"
#include "showir.h"
#include "messages.h"


sym	symtab[4096];		/* Symbol table */
sym	*symptr = &(symtab[0]);	/* Pointer to next free entry */
typ	curtyp;			/* Current type */

int	curscope = 0;		/* Current scope depth */


static	int	serialstack[256] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};

static	int	curserial = 0;


sym *
enter(register char *s, int used)
{
	register char *p = ((char *) malloc(strlen(s) + 1));

	strcpy(p, s);
	symptr->type = curtyp;
	symptr->text = p;
	symptr->serial = serialstack[curscope];
	symptr->scope = curscope;
	symptr->decline = zzline-1;
	symptr->declared = SYM_DECLARED;
	symptr->used = used;
	if ((curtyp.dim == 0) &&
	    (curtyp.attr & (TYP_CHAR | TYP_SHORT | TYP_INT |
			    TYP_LONG | TYP_LLONG | TYP_SWAR))) {
		warn("zero-element array promoted to 1 element");
		symptr->type.dim = 1;
	}
	return(symptr++);
}

sym *
lookup(register char *s)
{
	register sym *p = (symptr - 1);

	while (p >= &(symtab[0])) {
		/* symbol table maintains ALL entries,
		   even entries for things that are not valid now,
		   but each is tagged with lexical serial and scope;
		   text must match and (serial, scope) must be valid
		*/
		if ((serialstack[p->scope] == p->serial) &&
		    (strcmp(p->text, s) == 0)) {
			return(p);
		}
		--p;
	}
	return(NULL);
}

sym *
thiscope(register char *s)
{
	register sym *p = lookup(s);

	if ( (p) && (p->scope==curscope) )
		return p;
	else
		return NULL;
}

void
newscope(void)
{
	serialstack[++curscope] = ++curserial;
}

void
endscope(void)
{
	serialstack[curscope] = -1;
	--curscope;
}

void
showunused(void)
{
	/* Walk back through symtab, announcing symbols in this scope and
	   serial which are not marked as used.
	*/
	register sym *p = (symptr - 1);

	while (p >= &(symtab[0])) {
		if ((serialstack[curscope] == p->serial) &&
		    (p->used == SYM_UNUSED))
		{
			fprint_curr_function ( stderr );
			fprintf ( stderr,
			  "%s:%d: warning: unused variable `%s'\n",
			  sourcename,
			  p->decline,
			  p->text);
		}
		--p;
	}
}

void
showunusedstatics(void)
{
	/* Walk back through symtab, announcing static symbols in this scope
	   and serial which are not marked as used.
	*/
	register sym *p = (symptr - 1);

	while (p >= &(symtab[0])) {
		if ((serialstack[curscope] == p->serial) &&
		    (p->type.attr & TYP_STATIC) &&
		    (p->used == SYM_UNUSED))
		{
			fprint_curr_function ( stderr );
			fprintf ( stderr,
			  "%s:%d: warning: unused variable `%s'\n",
			  sourcename,
			  p->decline,
			  p->text);
		}
		--p;
	}
}

#ifdef NOTDEFD
static void
p_scope(int serial)
{
	register sym *p = &(symtab[0]);

	while (p <= (symptr - 1)) {
		/* This serial number and non-empty text */
		if ((p->serial == serial) &&
		    (p->text != 0) &&
		    (*(p->text)) &&
		    (!(p->type.attr & TYP_LAB))) {
			fprintf(Cout, "\t");
			p_ctype(&(p->type));
			fprintf(Cout, "%s", p->text);
			p_cdim(&(p->type));
			fprintf(Cout, ";\n");
		}
		++p;
	}
}
#endif


sym *
undeclared(char *name)
{
	static int first=1;
	sym *symbol;

	if ( (symbol = lookup ( name )) == NULL )
	{
		fprint_curr_function ( stderr );
		fprintf ( stderr,
			  "%s:%d: `%s' undeclared (first use this function)\n",
			  sourcename,
			  zzline,
			  name );
		if ( first )
		{
			fprintf ( stderr,
				  "%s:%d: (Each undeclared identifier "
				  "is reported only once\n",
				  sourcename,
				  zzline );
			fprintf ( stderr, "%s:%d: for each function it "
				  "appears in.)\n", sourcename, zzline );
			first = 0;
		}
		symbol = enter ( name, SYM_USED );
		symbol->declared = SYM_UNDECLARED;
	}
	return symbol;
}

